/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.beam.sdk.io.pulsar;

import org.apache.beam.sdk.transforms.DoFn;
import org.apache.beam.sdk.transforms.SerializableFunction;
import org.apache.pulsar.client.api.CompressionType;
import org.apache.pulsar.client.api.Producer;
import org.apache.pulsar.client.api.PulsarClient;
import org.apache.pulsar.client.api.PulsarClientException;

/** DoFn for writing to Apache Pulsar. */
@SuppressWarnings({"nullness"})
public class WriteToPulsarDoFn extends DoFn<byte[], Void> {
  private final SerializableFunction<String, PulsarClient> clientFn;
  private transient Producer<byte[]> producer;
  private transient PulsarClient client;
  private String clientUrl;
  private String topic;

  WriteToPulsarDoFn(PulsarIO.Write transform) {
    this.clientUrl = transform.getClientUrl();
    this.topic = transform.getTopic();
    this.clientFn = transform.getPulsarClient();
  }

  @Setup
  public void setup() {
    if (client == null) {
      if (clientUrl == null) {
        clientUrl = PulsarIOUtils.LOCAL_SERVICE_URL;
      }
      client = clientFn.apply(clientUrl);
    }
  }

  @StartBundle
  public void startBundle() throws PulsarClientException {
    producer = client.newProducer().topic(topic).compressionType(CompressionType.LZ4).create();
  }

  @ProcessElement
  public void processElement(@Element byte[] messageToSend) throws Exception {
    producer.send(messageToSend);
  }

  @FinishBundle
  public void finishBundle() throws PulsarClientException {
    producer.close();
  }

  @Teardown
  public void teardown() throws PulsarClientException {
    client.close();
  }
}
