// Licensed to the Apache Software Foundation (ASF) under one or more
// contributor license agreements.  See the NOTICE file distributed with
// this work for additional information regarding copyright ownership.
// The ASF licenses this file to You under the Apache License, Version 2.0
// (the "License"); you may not use this file except in compliance with
// the License.  You may obtain a copy of the License at
//
//    http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// Package runnerlib contains utilities for submitting Go pipelines
// to a Beam model runner.
package runnerlib

import (
	"context"
	"fmt"
	"os"
	"os/exec"
	"path/filepath"
	"runtime"
	"strings"
	"time"

	"sync/atomic"

	"github.com/apache/beam/sdks/go/pkg/beam/internal/errors"
	"github.com/apache/beam/sdks/go/pkg/beam/log"
)

// IsWorkerCompatibleBinary returns the path to itself and true if running
// a linux-amd64 binary that can directly be used as a worker binary.
func IsWorkerCompatibleBinary() (string, bool) {
	return "", false
}

var unique int32

// BuildTempWorkerBinary creates a local worker binary in the tmp directory
// for linux/amd64. Caller responsible for deleting the binary.
func BuildTempWorkerBinary(ctx context.Context) (string, error) {
	id := atomic.AddInt32(&unique, 1)
	filename := filepath.Join(os.TempDir(), fmt.Sprintf("worker-%v-%v", id, time.Now().UnixNano()))
	if err := BuildWorkerBinary(ctx, filename); err != nil {
		return "", err
	}
	return filename, nil
}

// BuildWorkerBinary creates a local worker binary for linux/amd64. It finds the filename
// by examining the call stack. We want the user entry (*), for example:
//
//   /Users/herohde/go/src/github.com/apache/beam/sdks/go/pkg/beam/runners/beamexec/main.go (skip: 2)
// * /Users/herohde/go/src/github.com/apache/beam/sdks/go/examples/wordcount/wordcount.go (skip: 3)
//   /usr/local/go/src/runtime/proc.go (skip: 4)      // not always present
//   /usr/local/go/src/runtime/asm_amd64.s (skip: 4 or 5)
func BuildWorkerBinary(ctx context.Context, filename string) error {
	program := ""
	var isTest bool
	for i := 3; ; i++ {
		_, file, _, ok := runtime.Caller(i)
		if !ok || !strings.HasSuffix(file, ".go") || strings.HasSuffix(file, "runtime/proc.go") {
			break
		} else if strings.HasSuffix(file, "testing/testing.go") {
			isTest = true
			break
		}
		program = file
	}
	if !strings.HasSuffix(program, ".go") {
		return errors.New("could not detect user main")
	}

	log.Infof(ctx, "Cross-compiling %v as %v", program, filename)

	// Cross-compile given go program. Not awesome.
	var build []string
	if isTest {
		program = program[:strings.LastIndex(program, "/")+1]
		program = program + "."
		build = []string{"go", "test", "-c", "-o", filename, program}
	} else {
		build = []string{"go", "build", "-o", filename, program}
	}

	cmd := exec.Command(build[0], build[1:]...)
	cmd.Env = append(os.Environ(), "GOOS=linux", "GOARCH=amd64")
	if out, err := cmd.CombinedOutput(); err != nil {
		return errors.Errorf("failed to cross-compile %v: %v\n%v", program, err, string(out))
	}
	return nil
}
