/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.util.config.memory;

import org.apache.flink.configuration.ConfigOption;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.IllegalConfigurationException;
import org.apache.flink.configuration.MemorySize;
import org.apache.flink.runtime.util.config.memory.CommonProcessMemorySpec;
import org.apache.flink.runtime.util.config.memory.FlinkMemory;
import org.apache.flink.runtime.util.config.memory.FlinkMemoryUtils;
import org.apache.flink.runtime.util.config.memory.JvmMetaspaceAndOverhead;
import org.apache.flink.runtime.util.config.memory.ProcessMemoryOptions;
import org.apache.flink.runtime.util.config.memory.ProcessMemorySpec;
import org.apache.flink.runtime.util.config.memory.RangeFraction;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProcessMemoryUtils<FM extends FlinkMemory> {
    private static final Logger LOG = LoggerFactory.getLogger(ProcessMemoryUtils.class);
    private final ProcessMemoryOptions options;
    private final FlinkMemoryUtils<FM> flinkMemoryUtils;

    public ProcessMemoryUtils(ProcessMemoryOptions options, FlinkMemoryUtils<FM> flinkMemoryUtils) {
        this.options = Preconditions.checkNotNull(options);
        this.flinkMemoryUtils = Preconditions.checkNotNull(flinkMemoryUtils);
    }

    public CommonProcessMemorySpec<FM> memoryProcessSpecFromConfig(Configuration config) {
        if (this.options.getRequiredFineGrainedOptions().stream().allMatch(config::contains)) {
            return this.deriveProcessSpecWithExplicitInternalMemory(config);
        }
        if (config.contains(this.options.getTotalFlinkMemoryOption())) {
            return this.deriveProcessSpecWithTotalFlinkMemory(config);
        }
        if (config.contains(this.options.getTotalProcessMemoryOption())) {
            return this.deriveProcessSpecWithTotalProcessMemory(config);
        }
        return this.failBecauseRequiredOptionsNotConfigured();
    }

    private CommonProcessMemorySpec<FM> deriveProcessSpecWithExplicitInternalMemory(Configuration config) {
        FM flinkInternalMemory = this.flinkMemoryUtils.deriveFromRequiredFineGrainedOptions(config);
        JvmMetaspaceAndOverhead jvmMetaspaceAndOverhead = this.deriveJvmMetaspaceAndOverheadFromTotalFlinkMemory(config, flinkInternalMemory.getTotalFlinkMemorySize());
        return new CommonProcessMemorySpec<FM>(flinkInternalMemory, jvmMetaspaceAndOverhead);
    }

    private CommonProcessMemorySpec<FM> deriveProcessSpecWithTotalFlinkMemory(Configuration config) {
        MemorySize totalFlinkMemorySize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getTotalFlinkMemoryOption());
        FM flinkInternalMemory = this.flinkMemoryUtils.deriveFromTotalFlinkMemory(config, totalFlinkMemorySize);
        JvmMetaspaceAndOverhead jvmMetaspaceAndOverhead = this.deriveJvmMetaspaceAndOverheadFromTotalFlinkMemory(config, totalFlinkMemorySize);
        return new CommonProcessMemorySpec<FM>(flinkInternalMemory, jvmMetaspaceAndOverhead);
    }

    private CommonProcessMemorySpec<FM> deriveProcessSpecWithTotalProcessMemory(Configuration config) {
        MemorySize totalProcessMemorySize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getTotalProcessMemoryOption());
        JvmMetaspaceAndOverhead jvmMetaspaceAndOverhead = this.deriveJvmMetaspaceAndOverheadWithTotalProcessMemory(config, totalProcessMemorySize);
        MemorySize totalFlinkMemorySize = totalProcessMemorySize.subtract(jvmMetaspaceAndOverhead.getTotalJvmMetaspaceAndOverheadSize());
        FM flinkInternalMemory = this.flinkMemoryUtils.deriveFromTotalFlinkMemory(config, totalFlinkMemorySize);
        return new CommonProcessMemorySpec<FM>(flinkInternalMemory, jvmMetaspaceAndOverhead);
    }

    private CommonProcessMemorySpec<FM> failBecauseRequiredOptionsNotConfigured() {
        CharSequence[] internalMemoryOptionKeys = (String[])this.options.getRequiredFineGrainedOptions().stream().map(ConfigOption::key).toArray(String[]::new);
        throw new IllegalConfigurationException(String.format("Either required fine-grained memory (%s), or Total Flink Memory size (%s), or Total Process Memory size (%s) need to be configured explicitly.", String.join((CharSequence)" and ", internalMemoryOptionKeys), this.options.getTotalFlinkMemoryOption(), this.options.getTotalProcessMemoryOption()));
    }

    private JvmMetaspaceAndOverhead deriveJvmMetaspaceAndOverheadWithTotalProcessMemory(Configuration config, MemorySize totalProcessMemorySize) {
        MemorySize jvmOverheadSize;
        MemorySize jvmMetaspaceSize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getJvmOptions().getJvmMetaspaceOption());
        JvmMetaspaceAndOverhead jvmMetaspaceAndOverhead = new JvmMetaspaceAndOverhead(jvmMetaspaceSize, jvmOverheadSize = ProcessMemoryUtils.deriveWithFraction("jvm overhead memory", totalProcessMemorySize, this.getJvmOverheadRangeFraction(config)));
        if (jvmMetaspaceAndOverhead.getTotalJvmMetaspaceAndOverheadSize().getBytes() > totalProcessMemorySize.getBytes()) {
            throw new IllegalConfigurationException("Sum of configured JVM Metaspace (" + jvmMetaspaceAndOverhead.getMetaspace().toHumanReadableString() + ") and JVM Overhead (" + jvmMetaspaceAndOverhead.getOverhead().toHumanReadableString() + ") exceed configured Total Process Memory (" + totalProcessMemorySize.toHumanReadableString() + ").");
        }
        return jvmMetaspaceAndOverhead;
    }

    public JvmMetaspaceAndOverhead deriveJvmMetaspaceAndOverheadFromTotalFlinkMemory(Configuration config, MemorySize totalFlinkMemorySize) {
        JvmMetaspaceAndOverhead jvmMetaspaceAndOverhead;
        MemorySize jvmMetaspaceSize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getJvmOptions().getJvmMetaspaceOption());
        MemorySize totalFlinkAndJvmMetaspaceSize = totalFlinkMemorySize.add(jvmMetaspaceSize);
        if (config.contains(this.options.getTotalProcessMemoryOption())) {
            MemorySize jvmOverheadSize = this.deriveJvmOverheadFromTotalFlinkMemoryAndOtherComponents(config, totalFlinkMemorySize);
            jvmMetaspaceAndOverhead = new JvmMetaspaceAndOverhead(jvmMetaspaceSize, jvmOverheadSize);
        } else {
            MemorySize jvmOverheadSize = ProcessMemoryUtils.deriveWithInverseFraction("jvm overhead memory", totalFlinkAndJvmMetaspaceSize, this.getJvmOverheadRangeFraction(config));
            jvmMetaspaceAndOverhead = new JvmMetaspaceAndOverhead(jvmMetaspaceSize, jvmOverheadSize);
            this.sanityCheckTotalProcessMemory(config, totalFlinkMemorySize, jvmMetaspaceAndOverhead);
        }
        return jvmMetaspaceAndOverhead;
    }

    private MemorySize deriveJvmOverheadFromTotalFlinkMemoryAndOtherComponents(Configuration config, MemorySize totalFlinkMemorySize) {
        MemorySize totalProcessMemorySize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getTotalProcessMemoryOption());
        MemorySize jvmMetaspaceSize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getJvmOptions().getJvmMetaspaceOption());
        MemorySize totalFlinkAndJvmMetaspaceSize = totalFlinkMemorySize.add(jvmMetaspaceSize);
        if (totalProcessMemorySize.getBytes() < totalFlinkAndJvmMetaspaceSize.getBytes()) {
            throw new IllegalConfigurationException("The configured Total Process Memory size (%s) is less than the sum of the derived Total Flink Memory size (%s) and the configured or default JVM Metaspace size  (%s).", totalProcessMemorySize.toHumanReadableString(), totalFlinkMemorySize.toHumanReadableString(), jvmMetaspaceSize.toHumanReadableString());
        }
        MemorySize jvmOverheadSize = totalProcessMemorySize.subtract(totalFlinkAndJvmMetaspaceSize);
        this.sanityCheckJvmOverhead(config, jvmOverheadSize, totalProcessMemorySize);
        return jvmOverheadSize;
    }

    private void sanityCheckJvmOverhead(Configuration config, MemorySize derivedJvmOverheadSize, MemorySize totalProcessMemorySize) {
        RangeFraction jvmOverheadRangeFraction = this.getJvmOverheadRangeFraction(config);
        if (derivedJvmOverheadSize.getBytes() > jvmOverheadRangeFraction.getMaxSize().getBytes() || derivedJvmOverheadSize.getBytes() < jvmOverheadRangeFraction.getMinSize().getBytes()) {
            throw new IllegalConfigurationException("Derived JVM Overhead size (" + derivedJvmOverheadSize.toHumanReadableString() + ") is not in configured JVM Overhead range [" + jvmOverheadRangeFraction.getMinSize().toHumanReadableString() + ", " + jvmOverheadRangeFraction.getMaxSize().toHumanReadableString() + "].");
        }
        if (config.contains(this.options.getJvmOptions().getJvmOverheadFraction()) && !derivedJvmOverheadSize.equals(totalProcessMemorySize.multiply(jvmOverheadRangeFraction.getFraction()))) {
            LOG.info("The derived JVM Overhead size ({}) does not match the configured or default JVM Overhead fraction ({}) from the configured Total Process Memory size ({}). The derived JVM Overhead size will be used.", new Object[]{derivedJvmOverheadSize.toHumanReadableString(), jvmOverheadRangeFraction.getFraction(), totalProcessMemorySize.toHumanReadableString()});
        }
    }

    private void sanityCheckTotalProcessMemory(Configuration config, MemorySize totalFlinkMemory, JvmMetaspaceAndOverhead jvmMetaspaceAndOverhead) {
        MemorySize configuredTotalProcessMemorySize;
        MemorySize derivedTotalProcessMemorySize = totalFlinkMemory.add(jvmMetaspaceAndOverhead.getMetaspace()).add(jvmMetaspaceAndOverhead.getOverhead());
        if (config.contains(this.options.getTotalProcessMemoryOption()) && !(configuredTotalProcessMemorySize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getTotalProcessMemoryOption())).equals(derivedTotalProcessMemorySize)) {
            throw new IllegalConfigurationException(String.format("Configured and Derived memory sizes (total %s) do not add up to the configured Total Process Memory size (%s). Configured and Derived memory sizes are: Total Flink Memory (%s), JVM Metaspace (%s), JVM Overhead (%s).", derivedTotalProcessMemorySize.toHumanReadableString(), configuredTotalProcessMemorySize.toHumanReadableString(), totalFlinkMemory.toHumanReadableString(), jvmMetaspaceAndOverhead.getMetaspace().toHumanReadableString(), jvmMetaspaceAndOverhead.getOverhead().toHumanReadableString()));
        }
    }

    private RangeFraction getJvmOverheadRangeFraction(Configuration config) {
        MemorySize minSize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getJvmOptions().getJvmOverheadMin());
        MemorySize maxSize = ProcessMemoryUtils.getMemorySizeFromConfig(config, this.options.getJvmOptions().getJvmOverheadMax());
        return ProcessMemoryUtils.getRangeFraction(minSize, maxSize, this.options.getJvmOptions().getJvmOverheadFraction(), config);
    }

    public static MemorySize getMemorySizeFromConfig(Configuration config, ConfigOption<MemorySize> option) {
        try {
            return Preconditions.checkNotNull(config.get(option), "The memory option is not set and has no default value.");
        }
        catch (Throwable t) {
            throw new IllegalConfigurationException("Cannot read memory size from config option '" + option.key() + "'.", t);
        }
    }

    public static RangeFraction getRangeFraction(MemorySize minSize, MemorySize maxSize, ConfigOption<Float> fractionOption, Configuration config) {
        double fraction = config.getFloat(fractionOption);
        try {
            return new RangeFraction(minSize, maxSize, fraction);
        }
        catch (IllegalArgumentException e) {
            throw new IllegalConfigurationException(String.format("Inconsistently configured %s (%s) and its min (%s), max (%s) value", fractionOption, fraction, minSize.toHumanReadableString(), maxSize.toHumanReadableString()), e);
        }
    }

    public static MemorySize deriveWithFraction(String memoryDescription, MemorySize base, RangeFraction rangeFraction) {
        MemorySize relative = base.multiply(rangeFraction.getFraction());
        return ProcessMemoryUtils.capToMinMax(memoryDescription, relative, rangeFraction);
    }

    public static MemorySize deriveWithInverseFraction(String memoryDescription, MemorySize base, RangeFraction rangeFraction) {
        Preconditions.checkArgument(rangeFraction.getFraction() < 1.0);
        MemorySize relative = base.multiply(rangeFraction.getFraction() / (1.0 - rangeFraction.getFraction()));
        return ProcessMemoryUtils.capToMinMax(memoryDescription, relative, rangeFraction);
    }

    private static MemorySize capToMinMax(String memoryDescription, MemorySize relative, RangeFraction rangeFraction) {
        long size = relative.getBytes();
        if (size > rangeFraction.getMaxSize().getBytes()) {
            LOG.info("The derived from fraction {} ({}) is greater than its max value {}, max value will be used instead", new Object[]{memoryDescription, relative.toHumanReadableString(), rangeFraction.getMaxSize().toHumanReadableString()});
            size = rangeFraction.getMaxSize().getBytes();
        } else if (size < rangeFraction.getMinSize().getBytes()) {
            LOG.info("The derived from fraction {} ({}) is less than its min value {}, min value will be used instead", new Object[]{memoryDescription, relative.toHumanReadableString(), rangeFraction.getMinSize().toHumanReadableString()});
            size = rangeFraction.getMinSize().getBytes();
        }
        return new MemorySize(size);
    }

    public static String generateJvmParametersStr(ProcessMemorySpec processSpec) {
        return ProcessMemoryUtils.generateJvmParametersStr(processSpec, true);
    }

    public static String generateJvmParametersStr(ProcessMemorySpec processSpec, boolean enableDirectMemoryLimit) {
        StringBuilder jvmArgStr = new StringBuilder();
        jvmArgStr.append("-Xmx").append(processSpec.getJvmHeapMemorySize().getBytes());
        jvmArgStr.append(" -Xms").append(processSpec.getJvmHeapMemorySize().getBytes());
        if (enableDirectMemoryLimit) {
            jvmArgStr.append(" -XX:MaxDirectMemorySize=").append(processSpec.getJvmDirectMemorySize().getBytes());
        }
        jvmArgStr.append(" -XX:MaxMetaspaceSize=").append(processSpec.getJvmMetaspaceSize().getBytes());
        return jvmArgStr.toString();
    }
}

