/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils.streamhist;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.math.IntMath;
import java.math.RoundingMode;
import java.util.Arrays;
import java.util.stream.Collectors;
import org.apache.cassandra.utils.streamhist.HistogramDataConsumer;
import org.apache.cassandra.utils.streamhist.TombstoneHistogram;

public class StreamingTombstoneHistogramBuilder {
    private final DataHolder bin;
    private Spool spool;
    private final int roundSeconds;

    public StreamingTombstoneHistogramBuilder(int maxBinSize, int maxSpoolSize, int roundSeconds) {
        assert (maxBinSize > 0 && maxSpoolSize >= 0 && roundSeconds > 0) : "Invalid arguments: maxBinSize:" + maxBinSize + " maxSpoolSize:" + maxSpoolSize + " delta:" + roundSeconds;
        this.roundSeconds = roundSeconds;
        this.bin = new DataHolder(maxBinSize + 1, roundSeconds);
        this.spool = new Spool(maxSpoolSize);
    }

    public void update(int point) {
        this.update(point, 1);
    }

    public void update(int point, int value) {
        assert (this.spool != null) : "update is being called after releaseBuffers. This could be functionally okay, but this assertion is a canary to alert about unintended use before it is necessary.";
        point = StreamingTombstoneHistogramBuilder.ceilKey(point, this.roundSeconds);
        if (this.spool.capacity > 0) {
            if (!this.spool.tryAddOrAccumulate(point, value)) {
                this.flushHistogram();
                boolean success = this.spool.tryAddOrAccumulate(point, value);
                assert (success) : "Can not add value to spool";
            }
        } else {
            this.flushValue(point, value);
        }
    }

    public void flushHistogram() {
        Spool spool = this.spool;
        if (spool != null) {
            spool.forEach(this::flushValue);
            spool.clear();
        }
    }

    public void releaseBuffers() {
        this.flushHistogram();
        this.spool = null;
    }

    private void flushValue(int key, int spoolValue) {
        this.bin.addValue(key, spoolValue);
        if (this.bin.isFull()) {
            this.bin.mergeNearestPoints();
        }
    }

    public TombstoneHistogram build() {
        this.flushHistogram();
        return new TombstoneHistogram(this.bin);
    }

    private static int ceilKey(int point, int bucketSize) {
        int delta = point % bucketSize;
        if (delta == 0) {
            return point;
        }
        return StreamingTombstoneHistogramBuilder.saturatingCastToMaxDeletionTime((long)point + (long)bucketSize - (long)delta);
    }

    public static int saturatingCastToInt(long value) {
        return (int)(value > Integer.MAX_VALUE ? Integer.MAX_VALUE : value);
    }

    public static int saturatingCastToMaxDeletionTime(long value) {
        return value < 0L || value > 0x7FFFFFFEL ? 0x7FFFFFFE : (int)value;
    }

    static class Spool {
        final int[] points;
        final int[] values;
        final int capacity;
        int size;

        Spool(int requestedCapacity) {
            if (requestedCapacity < 0) {
                throw new IllegalArgumentException("Illegal capacity " + requestedCapacity);
            }
            this.capacity = this.getPowerOfTwoCapacity(requestedCapacity);
            this.points = new int[this.capacity * 2];
            this.values = new int[this.capacity * 2];
            this.clear();
        }

        private int getPowerOfTwoCapacity(int requestedCapacity) {
            return requestedCapacity == 0 ? 0 : IntMath.pow((int)2, (int)IntMath.log2((int)requestedCapacity, (RoundingMode)RoundingMode.CEILING));
        }

        void clear() {
            Arrays.fill(this.points, -1);
            this.size = 0;
        }

        boolean tryAddOrAccumulate(int point, int delta) {
            if (this.size > this.capacity) {
                return false;
            }
            int cell = this.capacity - 1 & this.hash(point);
            for (int attempt = 0; attempt < 100; ++attempt) {
                if (!this.tryCell(cell + attempt, point, delta)) continue;
                return true;
            }
            return false;
        }

        private int hash(int i) {
            long largePrime = 948701839L;
            return (int)((long)i * largePrime);
        }

        <E extends Exception> void forEach(HistogramDataConsumer<E> consumer) throws E {
            for (int i = 0; i < this.points.length; ++i) {
                if (this.points[i] == -1) continue;
                consumer.consume(this.points[i], this.values[i]);
            }
        }

        private boolean tryCell(int cell, int point, int delta) {
            assert (cell >= 0 && point >= 0 && delta >= 0) : "Invalid arguments: cell:" + cell + " point:" + point + " delta:" + delta;
            if (this.points[cell %= this.points.length] == -1) {
                this.points[cell] = point;
                this.values[cell] = delta;
                ++this.size;
                return true;
            }
            if (this.points[cell] == point) {
                this.values[cell] = StreamingTombstoneHistogramBuilder.saturatingCastToInt((long)this.values[cell] + (long)delta);
                return true;
            }
            return false;
        }

        public String toString() {
            StringBuilder sb = new StringBuilder();
            sb.append('[');
            for (int i = 0; i < this.points.length; ++i) {
                if (this.points[i] == -1) continue;
                if (sb.length() > 1) {
                    sb.append(", ");
                }
                sb.append('[').append(this.points[i]).append(',').append(this.values[i]).append(']');
            }
            sb.append(']');
            return sb.toString();
        }
    }

    static class DataHolder {
        private static final long EMPTY = Long.MAX_VALUE;
        private final long[] data;
        private final int roundSeconds;

        DataHolder(int maxCapacity, int roundSeconds) {
            this.data = new long[maxCapacity];
            Arrays.fill(this.data, Long.MAX_VALUE);
            this.roundSeconds = roundSeconds;
        }

        DataHolder(DataHolder holder) {
            this.data = Arrays.copyOf(holder.data, holder.data.length);
            this.roundSeconds = holder.roundSeconds;
        }

        @VisibleForTesting
        int getValue(int point) {
            long key = this.wrap(point, 0L);
            int index = Arrays.binarySearch(this.data, key);
            if (index < 0) {
                index = -index - 1;
            }
            if (index >= this.data.length) {
                return -1;
            }
            if (this.unwrapPoint(this.data[index]) != point) {
                return -2;
            }
            return this.unwrapValue(this.data[index]);
        }

        boolean addValue(int point, int delta) {
            long key = this.wrap(point, 0L);
            int index = Arrays.binarySearch(this.data, key);
            if (index < 0) {
                index = -index - 1;
                assert (index < this.data.length) : "No more space in array";
                if (this.unwrapPoint(this.data[index]) != point) {
                    assert (this.data[this.data.length - 1] == Long.MAX_VALUE) : "No more space in array";
                    System.arraycopy(this.data, index, this.data, index + 1, this.data.length - index - 1);
                    this.data[index] = this.wrap(point, delta);
                    return true;
                }
                this.data[index] = this.wrap(point, (long)this.unwrapValue(this.data[index]) + (long)delta);
            } else {
                this.data[index] = this.wrap(point, (long)this.unwrapValue(this.data[index]) + (long)delta);
            }
            return false;
        }

        @VisibleForTesting
        void mergeNearestPoints() {
            assert (this.isFull()) : "DataHolder must be full in order to merge two points";
            int[] smallestDifference = this.findPointPairWithSmallestDistance();
            int point1 = smallestDifference[0];
            int point2 = smallestDifference[1];
            long key = this.wrap(point1, 0L);
            int index = Arrays.binarySearch(this.data, key);
            if (index < 0) {
                index = -index - 1;
                assert (index < this.data.length) : "Not found in array";
                assert (this.unwrapPoint(this.data[index]) == point1) : "Not found in array";
            }
            long value1 = this.unwrapValue(this.data[index]);
            long value2 = this.unwrapValue(this.data[index + 1]);
            assert (this.unwrapPoint(this.data[index + 1]) == point2) : "point2 should follow point1";
            long sum = value1 + value2;
            int newPoint = StreamingTombstoneHistogramBuilder.saturatingCastToInt(((long)point1 * value1 + (long)point2 * value2) / sum);
            newPoint = StreamingTombstoneHistogramBuilder.ceilKey(newPoint, this.roundSeconds);
            this.data[index] = this.wrap(newPoint, StreamingTombstoneHistogramBuilder.saturatingCastToInt(sum));
            System.arraycopy(this.data, index + 2, this.data, index + 1, this.data.length - index - 2);
            this.data[this.data.length - 1] = Long.MAX_VALUE;
        }

        private int[] findPointPairWithSmallestDistance() {
            assert (this.isFull()) : "The DataHolder must be full in order to find the closest pair of points";
            int point1 = 0;
            int point2 = Integer.MAX_VALUE;
            for (int i = 0; i < this.data.length - 1; ++i) {
                int pointA = this.unwrapPoint(this.data[i]);
                int pointB = this.unwrapPoint(this.data[i + 1]);
                assert (pointB > pointA) : "DataHolder not sorted, p2(" + pointB + ") < p1(" + pointA + ") for " + this;
                if (point2 - point1 <= pointB - pointA) continue;
                point1 = pointA;
                point2 = pointB;
            }
            return new int[]{point1, point2};
        }

        private int[] unwrap(long key) {
            int point = this.unwrapPoint(key);
            int value = this.unwrapValue(key);
            return new int[]{point, value};
        }

        private int unwrapPoint(long key) {
            return (int)(key >> 32);
        }

        private int unwrapValue(long key) {
            return (int)(key & 0xFFFFFFFFL);
        }

        private long wrap(int point, long value) {
            assert (point >= 0) : "Invalid argument: point:" + point;
            return (long)point << 32 | (long)StreamingTombstoneHistogramBuilder.saturatingCastToInt(value);
        }

        public String toString() {
            return Arrays.stream(this.data).filter(x -> x != Long.MAX_VALUE).mapToObj(this::unwrap).map(Arrays::toString).collect(Collectors.joining());
        }

        public boolean isFull() {
            return this.data[this.data.length - 1] != Long.MAX_VALUE;
        }

        public <E extends Exception> void forEach(HistogramDataConsumer<E> histogramDataConsumer) throws E {
            for (long datum : this.data) {
                if (datum == Long.MAX_VALUE) break;
                histogramDataConsumer.consume(this.unwrapPoint(datum), this.unwrapValue(datum));
            }
        }

        public int size() {
            int[] accumulator = new int[1];
            this.forEach((point, value) -> {
                accumulator[0] = accumulator[0] + 1;
            });
            return accumulator[0];
        }

        public double sum(int b) {
            long pointAndValue;
            double sum = 0.0;
            for (int i = 0; i < this.data.length && (pointAndValue = this.data[i]) != Long.MAX_VALUE; ++i) {
                int point = this.unwrapPoint(pointAndValue);
                int value = this.unwrapValue(pointAndValue);
                if (point > b) {
                    if (i == 0) {
                        return 0.0;
                    }
                    int prevPoint = this.unwrapPoint(this.data[i - 1]);
                    int prevValue = this.unwrapValue(this.data[i - 1]);
                    double weight = (double)(b - prevPoint) / (double)(point - prevPoint);
                    double mb = (double)prevValue + (double)(value - prevValue) * weight;
                    sum -= (double)prevValue;
                    sum += ((double)prevValue + mb) * weight / 2.0;
                    return sum += (double)prevValue / 2.0;
                }
                sum += (double)value;
            }
            return sum;
        }

        public int hashCode() {
            return Arrays.hashCode(this.data);
        }

        public boolean equals(Object o) {
            if (!(o instanceof DataHolder)) {
                return false;
            }
            DataHolder other = (DataHolder)o;
            if (this.size() != other.size()) {
                return false;
            }
            for (int i = 0; i < this.size(); ++i) {
                if (this.data[i] == other.data[i]) continue;
                return false;
            }
            return true;
        }
    }
}

