/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.iotdb.schema;

import org.apache.tsfile.exception.PathParseException;
import org.apache.tsfile.read.common.parser.PathNodesGenerator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * Before creating paths in IoTDB, it's essential to check whether the paths are correct to avoid
 * errors.
 *
 * <p>This example checks the paths . You can add paths to the inputList or read paths from files
 * generated by export-csv.sh/export-csv.bat.
 *
 * <p>The usage of export-csv.sh can be found in the <a
 * href="https://iotdb.apache.org/zh/UserGuide/V1.2.x/Tools-System/Import-Export-Tool.html#csv%E5%AF%BC%E5%85%A5%E5%AF%BC%E5%87%BA%E5%B7%A5%E5%85%B7">documentation</a>.
 * For example: ./export-csv.sh -h 127.0.0.1 -p 6667 -u root -pw root -td . -q "show timeseries"
 */
public class PathCheckExample {
  private static final List<String> INPUT_LIST = new ArrayList<>();
  private static final String DIR = "/Users/root/iotdb/tools";

  private static final Logger LOGGER = LoggerFactory.getLogger(PathCheckExample.class);

  // concurrent thread of path check
  private static final int CONCURRENCY = 5;

  public static void main(String[] args) {
    batchCheck();
    dirCheck();
  }

  private static void batchCheck() {
    INPUT_LIST.add("root.test.d1.s1");
    INPUT_LIST.add("root.b+.d1.s2");
    INPUT_LIST.add("root.test.1.s3");
    INPUT_LIST.add("root.test.d-j.s4");
    INPUT_LIST.add("root.test.'8`7'.s5");
    INPUT_LIST.add("root.test.`1`.s6");
    INPUT_LIST.add("root.test.\"d+b\".s7");
    for (String path : INPUT_LIST) {
      checkPath(path);
    }
  }

  // This function wile check whether the paths are correct in current version.
  private static void checkPath(String path) {
    try {
      PathNodesGenerator.checkPath(path);
    } catch (PathParseException e) {
      LOGGER.error("{} is not a legal path.", path);
    }
  }

  /**
   * Using multiple threads to check the paths in the files of the directory. the files are created
   * by export-csv.sh/export-csv.bat.
   */
  public static void dirCheck() {

    List<Future<Void>> futureList = new ArrayList<>();
    ExecutorService executorService = Executors.newFixedThreadPool(CONCURRENCY);
    File dir = new File(DIR);
    for (File file : Objects.requireNonNull(dir.listFiles())) {
      if (file.getName().startsWith("dump") && file.getName().endsWith(".csv")) {
        Future<Void> future = executorService.submit(new CheckThread(file));
        futureList.add(future);
      }
    }
    try {
      for (Future<Void> future : futureList) {
        future.get();
      }
    } catch (InterruptedException | ExecutionException e) {
      LOGGER.error("Error when checking paths.");
      Thread.currentThread().interrupt();
    }
    executorService.shutdown();
  }

  static class CheckThread implements Callable<Void> {
    File file;

    public CheckThread(File file) {
      this.file = file;
    }

    @Override
    public Void call() {
      try (BufferedReader br = new BufferedReader(new FileReader(file))) {

        String line = br.readLine();
        while ((line = br.readLine()) != null) {
          String[] split = line.split(",");
          String path = split[0];
          checkPath(path);
        }
      } catch (IOException e) {
        LOGGER.error("Error reading file: {}", file.getName());
        throw new RuntimeException(e);
      }
      return null;
    }
  }
}
