/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.viewController;

/**
 * Each executor configured with the constructor will be asked to execute the
 * requested method on the bean. The search will end if the executor returns "true".
 * <p>
 * As an example, this can be configured to first try an executor that checks whether
 * the bean implements a specific interface, then try one that uses reflection to find
 * magic method-names, then try one that looks for annotations.
 */
public class CompositeViewControllerExecutor implements ViewControllerExecutor
{
    private final ViewControllerExecutor[] viewControllerExecutors;

    public CompositeViewControllerExecutor(ViewControllerExecutor[] viewControllerExecutors)
    {
        this.viewControllerExecutors = new ViewControllerExecutor[viewControllerExecutors.length];
        System.arraycopy(viewControllerExecutors, 0, this.viewControllerExecutors, 0, viewControllerExecutors.length);
    }


    public boolean invokeInitView(String beanName, Object bean)
    {
        for (int i = 0; i< viewControllerExecutors.length; i++)
        {
            if (viewControllerExecutors[i].invokeInitView(beanName, bean))
            {
                return true;
            }
        }

        return false;
    }

    public boolean invokePreRenderView(String beanName, Object bean)
    {
        for (int i = 0; i< viewControllerExecutors.length; i++)
        {
            if (viewControllerExecutors[i].invokePreRenderView(beanName, bean))
            {
                return true;
            }
        }

        return false;
    }

    public boolean invokePreProcess(String beanName, Object bean)
    {
        for (int i = 0; i< viewControllerExecutors.length; i++)
        {
            if (viewControllerExecutors[i].invokePreProcess(beanName, bean))
            {
                return true;
            }
        }

        return false;
    }
}