/* Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 **/

package org.apache.myfaces.portlet.faces.application;

import com.sun.faces.application.ViewHandlerResponseWrapper;

import java.io.ByteArrayOutputStream;
import java.io.CharArrayWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;

import java.nio.ByteBuffer;

import javax.faces.context.ExternalContext;

import javax.faces.context.FacesContext;

import javax.portlet.faces.Bridge;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletResponseWrapper;

public class BridgeMojarraRenderFilter
  implements Filter
{
  public BridgeMojarraRenderFilter()
  {
  }

  public void init(FilterConfig config)
  {

  }

  public void doFilter(ServletRequest request, ServletResponse response, 
                       FilterChain chain)
    throws IOException, ServletException
  {
    // only do any work if attribute set indicates we should
    Boolean renderAfter = 
      (Boolean) request.getAttribute(Bridge.RENDER_CONTENT_AFTER_VIEW);
    if (renderAfter == null || renderAfter == Boolean.FALSE)
    {
      chain.doFilter(request, response);
      return;
    }

    // otherwise try and support render after
    BridgeRenderFilterResponseWrapper wrapped = 
      new BridgeRenderFilterResponseWrapper((HttpServletResponse) response);
    
    // temporarily set as the response object in the ExternalContext -- as the Mojarra ViewTag
    // gets this instance to verify it can write after.
    ExternalContext extCtx = FacesContext.getCurrentInstance().getExternalContext();
    Object currentResponse = extCtx.getResponse();
    extCtx.setResponse(wrapped);
    
    chain.doFilter(request, wrapped);
    
    // reset the ExternalContext response
    extCtx.setResponse(currentResponse);

    // wrap the response as a JSF RI wrapped response
    // execute the chain
    // Follow the JSTL 1.2 spec, section 7.4,  
    // on handling status codes on a forward
    if (wrapped.getStatus() < 200 || wrapped.getStatus() > 299)
    {
      // flush the contents of the wrapper to the response
      // this is necessary as the user may be using a custom 
      // error page - this content should be propagated
      if (wrapped.isChars())
      {
        PrintWriter writer = response.getWriter();
        writer.print(wrapped.getChars());
        writer.flush();
      }
      else
      {
        ServletOutputStream stream = response.getOutputStream();
        stream.write(wrapped.getBytes());
        stream.flush();
      }
    }
    else
    {
      // Put the AFTER_VIEW_CONTENT into request scope temporarily
      // Check both bytes and chars as we don't know how the dispatchee
      // deals with the Output.
      request.setAttribute(Bridge.AFTER_VIEW_CONTENT, 
                           (wrapped.isChars())? (Object) wrapped.getChars(): 
                           wrapped.getBytes());
    }
  }

  public void destroy()
  {

  }

  // Inner classes -- implements buffered response wrapper


  /**
   * <p>This class is used by {@link javax.faces.application.ViewHandler#createView} to obtain the
   * text that exists after the &lt;f:view&gt; tag.</p>
   */
  public class BridgeRenderFilterResponseWrapper
    extends ViewHandlerResponseWrapper
  {

    private DirectByteArrayServletOutputStream mByteStream;
    private CharArrayWriter mCharWriter;
    private PrintWriter mPrintWriter;
    private int mStatus = HttpServletResponse.SC_OK;


    public BridgeRenderFilterResponseWrapper(HttpServletResponse wrapped)
    {
      super(wrapped);
    }

    public void flushBuffer()
    {
      if (isChars())
      {
        mPrintWriter.flush();
      }
    }

    public int getBufferSize()
    {
      if (isBytes())
      {
        return mByteStream.size();
      }
      else
      {
        return mCharWriter.size();
      }
    }


    public void reset()
    {
      super.reset();
      if (isBytes())
      {
        mByteStream.reset();
      }
      else
      {
        mPrintWriter.flush();
        mCharWriter.reset();
      }
    }

    public void resetBuffer()
    {
      super.resetBuffer();
      if (isBytes())
      {
        mByteStream.reset();
      }
      else
      {
        mPrintWriter.flush();
        mCharWriter.reset();
      }
    }


    public ServletOutputStream getOutputStream()
      throws IOException
    {
      if (mPrintWriter != null)
      {
        throw new IllegalStateException();
      }
      if (mByteStream == null)
      {
        mByteStream = new DirectByteArrayServletOutputStream();
      }
      return mByteStream;
    }

    public PrintWriter getWriter()
      throws IOException
    {
      if (mByteStream != null)
      {
        throw new IllegalStateException();
      }
      if (mPrintWriter == null)
      {
        mCharWriter = new CharArrayWriter(4096);
        mPrintWriter = new PrintWriter(mCharWriter);
      }

      return mPrintWriter;
    }

    @Override
    public void sendError(int status, String msg)
      throws IOException
    {
      super.sendError(status, msg);
      // preserve status so can reply to getStatus()
      mStatus = status;
    }

    @Override
    public void sendError(int status)
      throws IOException
    {
      super.sendError(status);
      // preserve status so can reply to getStatus()
      mStatus = status;
    }

    @Override
    public void setStatus(int sc)
    {
      super.setStatus(sc);
      // preserve status so can reply to getStatus()
      mStatus = sc;
    }

    @Override
    public void setStatus(int sc, String sm)
    {
      super.setStatus(sc, sm);
      // preserve status so can reply to getStatus()
      mStatus = sc;
    }


    public int getStatus()
    {
      return mStatus;
    }

    public boolean isBytes()
    {
      return (mByteStream != null);
    }

    public boolean isChars()
    {
      return (mCharWriter != null);
    }

    public byte[] getBytes()
    {
      if (isBytes())
      {
        return mByteStream.toByteArray();
      }
      else
      {
        return null;
      }
    }

    public char[] getChars()
    {
      if (isChars())
      {
        mCharWriter.flush();
        return mCharWriter.toCharArray();
      }
      else
      {
        return null;
      }
    }
    
    public String toString()
    {
      if (isChars())
      {
        mCharWriter.flush();
        return mCharWriter.toString();
      }
      else
      {
        return mByteStream.toString();
      }
    }
    
    public void clearWrappedResponse() throws IOException {
      resetBuffers();
    }

    /**
     * Flush the current buffered content to the wrapped
     * response (this could be a Servlet or Portlet response)
     * @throws IOException if content cannot be written
     */
    public void flushContentToWrappedResponse()
      throws IOException
    {
      ServletResponse response = getResponse();
      
      flushBuffer();
      
      if (isBytes())
      {
        response.getOutputStream().write(getBytes());
        mByteStream.reset();
      }
      else
      {
        response.getWriter().write(getChars());
        mCharWriter.reset();
      }
      
    }

    /**
     * Flush the current buffered content to the provided <code>Writer</code>
     * @param writer target <code>Writer</code>
     * @param encoding the encoding that should be used
     * @throws IOException if content cannot be written
     */
    public void flushToWriter(Writer writer, String encoding)
      throws IOException
    {     
      flushBuffer();
      
      if (isBytes())
      {
        throw new IOException("Invalid flushToWriter as the code is writing bytes to an OutputStream.");
      }
      else
      {
        writer.write(getChars());
        mCharWriter.reset();
      }
    }

    /**
     * Clear the internal buffers.
     * @throws IOException if some odd error occurs
     */
    public void resetBuffers()
      throws IOException
    {
      if (isBytes())
      {
        mByteStream.reset();
      }
      else
      {
        mPrintWriter.flush();
        mCharWriter.reset();
      }
    }

  }


  // ----------------------------------------------------------- Inner Classes


  private class DirectByteArrayServletOutputStream
    extends ServletOutputStream
  {
    private DirectByteArrayOutputStream mByteArrayOutputStream;

    public DirectByteArrayServletOutputStream()
    {
      mByteArrayOutputStream = new DirectByteArrayOutputStream(4096);
    }

    public void write(int n)
    {
      mByteArrayOutputStream.write(n);
    }

    public byte[] toByteArray()
    {
      return mByteArrayOutputStream.toByteArray();
    }
    
    public int size()
    {
      return mByteArrayOutputStream.size();
    }
    
    public void reset()
    {
      mByteArrayOutputStream.reset();
    }

  }


  private class DirectByteArrayOutputStream
    extends ByteArrayOutputStream
  {


    // -------------------------------------------------------- Constructors


    public DirectByteArrayOutputStream(int initialCapacity)
    {
      super(initialCapacity);
    }


    // ------------------------------------------------------- PublicMethods


    /**
     * Return the buffer backing this ByteArrayOutputStream as a 
     * ByteBuffer.
     * @return buf wrapped in a ByteBuffer
     */
    public ByteBuffer getByteBuffer()
    {
      return (ByteBuffer.wrap(buf, 0, count));
    }

  }


  // end of class BridgeRenderFilterResponseWrapper
}
