/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.netbeans.modules.form;

import javax.swing.DefaultListModel;

import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.awt.Mnemonics;
import org.openide.util.HelpCtx;

/**
 *
 * @author  Pavel Buzek
 */
public class EventCustomEditor extends javax.swing.JPanel {

    static final long serialVersionUID =-4825059521634962952L;

    /** Creates new form EventCustomEditor */
    EventCustomEditor(EventProperty eventProperty) {
        this.eventProperty = eventProperty;

        initComponents();
        enableButtons();

        Mnemonics.setLocalizedText(addButton, FormUtils.getBundleString("CTL_EE_ADD")); // NOI18N
        Mnemonics.setLocalizedText(removeButton, FormUtils.getBundleString("CTL_EE_REMOVE")); // NOI18N
        Mnemonics.setLocalizedText(editButton, FormUtils.getBundleString("CTL_EE_RENAME")); // NOI18N
        Mnemonics.setLocalizedText(handlersListLabel, FormUtils.getBundleString("CTL_EE_Handlers")); // NOI18N

        handlersList.getAccessibleContext().setAccessibleDescription(
            FormUtils.getBundleString("ACSD_CTL_EE_Handlers")); // NOI18N
        addButton.getAccessibleContext().setAccessibleDescription(
            FormUtils.getBundleString("ACSD_CTL_EE_ADD")); // NOI18N
        removeButton.getAccessibleContext().setAccessibleDescription(
            FormUtils.getBundleString("ACSD_CTL_EE_REMOVE")); // NOI18N
        editButton.getAccessibleContext().setAccessibleDescription(
            FormUtils.getBundleString("ACSD_CTL_EE_RENAME")); // NOI18N
        getAccessibleContext().setAccessibleDescription(
            FormUtils.getBundleString("ACSD_EventCustomEditor")); // NOI18N

        HelpCtx.setHelpIDString(this, "gui.csh.handlers"); // NOI18N
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the FormEditor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        editButton = new javax.swing.JButton();
        removeButton = new javax.swing.JButton();
        addButton = new javax.swing.JButton();
        jScrollPane1 = new javax.swing.JScrollPane();
        String[] handlers = eventProperty.getEventHandlers();
        for (int i=0; i < handlers.length; i++) {
            handlersModel.addElement(handlers[i]);
        }
        handlersList = new javax.swing.JList();
        handlersList.setModel(handlersModel);
        if (handlers.length > 0) {
            handlersList.setSelectedIndex(0);
        }
        handlersListLabel = new javax.swing.JLabel();

        editButton.setText("Rename...");
        editButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                editButtonActionPerformed(evt);
            }
        });

        removeButton.setText("Remove");
        removeButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                removeButtonActionPerformed(evt);
            }
        });

        addButton.setText("Add...");
        addButton.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                addButtonActionPerformed(evt);
            }
        });

        handlersList.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
            public void valueChanged(javax.swing.event.ListSelectionEvent evt) {
                handlersListValueChanged(evt);
            }
        });
        jScrollPane1.setViewportView(handlersList);

        handlersListLabel.setLabelFor(handlersList);
        handlersListLabel.setText("Handlers");

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 189, Short.MAX_VALUE)
                    .addComponent(handlersListLabel, javax.swing.GroupLayout.DEFAULT_SIZE, 189, Short.MAX_VALUE))
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                    .addComponent(editButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(removeButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                    .addComponent(addButton, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE))
                .addContainerGap())
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
            .addGroup(layout.createSequentialGroup()
                .addContainerGap()
                .addComponent(handlersListLabel)
                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                    .addGroup(layout.createSequentialGroup()
                        .addComponent(addButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(removeButton)
                        .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                        .addComponent(editButton)
                        .addContainerGap(188, Short.MAX_VALUE))
                    .addComponent(jScrollPane1, javax.swing.GroupLayout.DEFAULT_SIZE, 275, Short.MAX_VALUE)))
        );
    }// </editor-fold>//GEN-END:initComponents

    private void handlersListValueChanged(javax.swing.event.ListSelectionEvent evt) {//GEN-FIRST:event_handlersListValueChanged
        enableButtons();
    }//GEN-LAST:event_handlersListValueChanged

    private void enableButtons() {
        if (handlersList.isSelectionEmpty()) {
            removeButton.setEnabled(false);
        } else {
            removeButton.setEnabled(true);
        }
        editButton.setEnabled(handlersList.getSelectedIndices().length == 1);
    }

    private void editButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_editButtonActionPerformed
        // Add your handling code here:
        int i = handlersList.getSelectedIndex();
        if (i >= 0) {
            String oldName = (String) handlersModel.get(i);
            NotifyDescriptor.InputLine nd = new NotifyDescriptor.InputLine(
                FormUtils.getBundleString("CTL_EE_RENAME_LABEL"), // NOI18N
                FormUtils.getBundleString("CTL_EE_RENAME_CAPTION")); // NOI18N
            nd.setInputText(oldName);

            if (DialogDisplayer.getDefault().notify(nd).equals(NotifyDescriptor.OK_OPTION)) {
                String newName = nd.getInputText();
                if (newName.equals(oldName)) return; // no change

                if (!org.openide.util.Utilities.isJavaIdentifier(newName)) { // invalid name
                    NotifyDescriptor.Message msg = new NotifyDescriptor.Message(
                        FormUtils.getBundleString("CTL_EE_NOT_IDENTIFIER"), // NOI18N
                        NotifyDescriptor.ERROR_MESSAGE);
                    DialogDisplayer.getDefault().notify(msg);
                    return;
                }

                if (handlersModel.indexOf(newName) >= 0) { // already exists
                    NotifyDescriptor.Message msg = new NotifyDescriptor.Message(
                        FormUtils.getBundleString("CTL_EE_ALREADY_EXIST"), // NOI18N
                        NotifyDescriptor.INFORMATION_MESSAGE);
                    DialogDisplayer.getDefault().notify(msg);
                    return;
                }

                int ii = changes.getAdded().indexOf(oldName);
                if (ii >= 0) { // a newly added handler was renamed
                    changes.getAdded().set(ii,newName);
                }
                else {
                    ii = changes.getRenamedNewNames().indexOf(oldName);
                    if (ii >= 0) // this handler has been already renamed
                        changes.getRenamedNewNames().set(ii, newName);
                    else {
                        changes.getRenamedOldNames().add(oldName);
                        changes.getRenamedNewNames().add(newName);
                    }
                }

                handlersModel.set(i,newName);
                handlersList.setSelectedIndex(i);
                enableButtons();
            }
        }
    }//GEN-LAST:event_editButtonActionPerformed

    private void removeButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_removeButtonActionPerformed
        Object[] handlers = handlersList.getSelectedValues();
        for (int i=0; i < handlers.length; i++) {
            int ii = changes.getAdded().indexOf(handlers[i]);
            if (ii >= 0) { // the handler was previously added - cancel it
                changes.getAdded().remove(ii);
            }
            else {
                ii = changes.getRenamedNewNames().indexOf(handlers[i]);
                String toRemove;
                if (ii >= 0) { // the handler was previously renamed - cancel it
                    changes.getRenamedNewNames().remove(ii);
                    toRemove = changes.getRenamedOldNames().get(ii);
                    changes.getRenamedOldNames().remove(ii);
                }
                else toRemove = (String) handlers[i];

                changes.getRemoved().add(toRemove);
            }
            handlersModel.removeElement(handlers[i]);
            enableButtons();
        }
    }//GEN-LAST:event_removeButtonActionPerformed

    private void addButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_addButtonActionPerformed
        NotifyDescriptor.InputLine nd = new NotifyDescriptor.InputLine(
            FormUtils.getBundleString("CTL_EE_ADD_LABEL"), // NOI18N
            FormUtils.getBundleString("CTL_EE_ADD_CAPTION")); // NOI18N
        if (DialogDisplayer.getDefault().notify(nd).equals(NotifyDescriptor.OK_OPTION)) {
            String newHandler = nd.getInputText();
            if (!org.openide.util.Utilities.isJavaIdentifier(newHandler)) {
                NotifyDescriptor.Message msg = new NotifyDescriptor.Message(
                    FormUtils.getBundleString("CTL_EE_NOT_IDENTIFIER"), // NOI18N
                    NotifyDescriptor.ERROR_MESSAGE);
                DialogDisplayer.getDefault().notify(msg);
                return;
            }

            if (handlersModel.indexOf(newHandler) >= 0) {
                NotifyDescriptor.Message msg = new NotifyDescriptor.Message(
                    FormUtils.getBundleString("CTL_EE_ALREADY_EXIST"), // NOI18N
                    NotifyDescriptor.INFORMATION_MESSAGE);
                DialogDisplayer.getDefault().notify(msg);
                return;
            }

            int ir = changes.getRemoved().indexOf(newHandler);
            if (ir >= 0) {
                changes.getRemoved().remove(ir);
            }
            else {
                changes.getAdded().add(newHandler);
            }
            handlersModel.addElement(newHandler);
            handlersList.setSelectedIndex(handlersModel.size() - 1);
            enableButtons();
        }
    }//GEN-LAST:event_addButtonActionPerformed

    public void doChanges() {
        try {
            eventProperty.setValue(changes);
        } catch (Exception e) { // should not happen
            e.printStackTrace();
        }
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton addButton;
    private javax.swing.JButton editButton;
    private javax.swing.JList handlersList;
    private javax.swing.JLabel handlersListLabel;
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JButton removeButton;
    // End of variables declaration//GEN-END:variables

    EventProperty eventProperty;
    DefaultListModel handlersModel = new DefaultListModel();
    EventProperty.Change changes = new EventProperty.Change();
}
