/*
 * tkUnixFontOS2.c --
 *
 *	Contains the OS2 implementation of the platform-independant
 *	font package interface.
 *
 * Copyright (c) 1996 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * RCS: @(#) $Id: tkUnixFontOS2.c,v 1.1 2000/10/03 08:09:30 ilya Exp ilya $
 */
 
#include "tkPort.h"
#include "tkInt.h"
#include "tkOS2Int.h"

#include "tkFont.h"

#define MAX_NUM_LFONTS	256		/* System limit (per process?) */

#ifndef ABS
#define ABS(n)	(((n) < 0) ? -(n) : (n))
#endif

/*
 * The following structure represents Unix's implementation of a font.
 */
 
typedef struct OS2Font {
    TkFont font;		/* Stuff used by generic font package.  Must
				 * be first in structure. */
    Display *display;		/* The display to which font belongs. */
    XFontStruct *fontStructPtr;	/* X information about font. */
    char types[256];		/* Array giving types of all characters in
				 * the font, used when displaying control
				 * characters.  See below for definition. */
    int widths[256];		/* Array giving widths of all possible
				 * characters in the font. */
    int underlinePos;		/* Offset from baseline to origin of
				 * underline bar (used for simulating a native
				 * underlined font). */
    int barHeight;		/* Height of underline or overstrike bar
				 * (used for simulating a native underlined or
				 * strikeout font). */
} OS2Font;

/*
 * Possible values for entries in the "types" field in a OS2Font structure,
 * which classifies the types of all characters in the given font.  This
 * information is used when measuring and displaying characters.
 *
 * NORMAL:		Standard character.
 * REPLACE:		This character doesn't print:  instead of
 *			displaying character, display a replacement
 *			sequence like "\n" (for those characters where
 *			ANSI C defines such a sequence) or a sequence
 *			of the form "\xdd" where dd is the hex equivalent
 *			of the character.
 * SKIP:		Don't display anything for this character.  This
 *			is only used where the font doesn't contain
 *			all the characters needed to generate
 *			replacement sequences.
 */ 

#define NORMAL		0
#define REPLACE		1
#define SKIP		2

/*
 * Characters used when displaying control sequences.
 */

static char hexChars[] = "0123456789abcdefxtnvr\\";

/*
 * The following table maps some control characters to sequences like '\n'
 * rather than '\x10'.  A zero entry in the table means no such mapping
 * exists, and the table only maps characters less than 0x10.
 */

static char mapChars[] = {
    0, 0, 0, 0, 0, 0, 0,
    'a', 'b', 't', 'n', 'v', 'f', 'r',
    0
};


static OS2Font *	AllocFont _ANSI_ARGS_((TkFont *tkFontPtr,
			    Tk_Window tkwin, XFontStruct *fontStructPtr,
			    CONST char *fontName));
static void		DrawChars _ANSI_ARGS_((Display *display,
			    Drawable drawable, GC gc, OS2Font *fontPtr,
			    CONST char *source, int numChars, int x,
			    int y));
static int		GetControlCharSubst _ANSI_ARGS_((int c, char buf[4]));

static XFontStruct *	OS2QueryFont(LONG font_ID);


PFONTMETRICS
fetchOS2fonts(CONST char *fontname, int *nptr, int isfamily)
{
	LONG reqFonts = 0L;
	CONST char *qname = (isfamily ? NULL : fontname);
	LONG remFonts = GpiQueryFonts(globalPS, QF_PUBLIC, qname, &reqFonts,
				      (LONG) sizeof(FONTMETRICS), NULL);
	PFONTMETRICS os2fonts;

	*nptr = remFonts;
#ifdef DEBUG
	if (remFonts == GPI_ALTERROR)
	    printf("    GpiQueryFonts %s ERROR %x\n", qname,
		   WinGetLastError(hab));
	else
	    printf("...%ld fonts '%s' found.\n",
		   (long)remFonts, fontname);
#endif
	if (!remFonts)
	    return NULL;
	/* Allocate space for the fonts */
	os2fonts = (PFONTMETRICS) ckalloc(remFonts * sizeof(FONTMETRICS));
	if (os2fonts == NULL) {
	    *nptr = 0;
	    return NULL;
	}

	/* Get the fonts that apply */
	reqFonts = remFonts;
	remFonts = GpiQueryFonts(globalPS, QF_PUBLIC, qname, &reqFonts,
				 (LONG) sizeof(FONTMETRICS), os2fonts);
#ifdef DEBUG
	if (remFonts == GPI_ALTERROR)
	    printf("    GpiQueryFonts %s ERROR %x\n", fontname,
		   WinGetLastError(hab));
	else
	    printf("    nr.of fonts [%s]: %d (%d remaining)\n",
		   fontname, reqFonts, remFonts);
#endif
	if (!fontname)
	    isfamily = 0;
	if (remFonts != GPI_ALTERROR && !isfamily)
	    return os2fonts;
	else if (remFonts != GPI_ALTERROR && *nptr) { /* Filter family */
	    PFONTMETRICS curFontPtr = os2fonts, lim = os2fonts + *nptr;
	    PFONTMETRICS lastFontPtr = os2fonts;
	    char buff[256];		/* Atom names cannot be longer... */
	    int useAtom = (strlen(fontname) >= FACESIZE), rc;
	    HATOMTBL      hatomtbl;

	    if (useAtom) {
		hatomtbl = WinQuerySystemAtomTable(); 
#ifdef DEBUG
		if (!hatomtbl)
		    printf(" WinQuerySystemAtomTable ERROR %x\n", 
			   WinGetLastError(hab));
#endif
	    }
	    while (curFontPtr < lim) {
		char *fam = curFontPtr->szFamilyname;

		/* Cannot just compare with the atom for familyname,
		   since atoms are case-sensitive... */
		if ((curFontPtr->fsType & (FM_TYPE_FACETRUNC|FM_TYPE_ATOMS))
		    == (FM_TYPE_FACETRUNC|FM_TYPE_ATOMS)) {
		    rc = WinQueryAtomName(hatomtbl, curFontPtr->FamilyNameAtom,
					  buff, sizeof(buff));
		    if (rc != 0)
			fam = buff;
		    else {
#ifdef DEBUG
			printf(" WinQueryAtomName ERROR %x\n", WinGetLastError(hab));
#endif
			goto next_font;
		    }
		} else if (useAtom)
		    goto next_font;

		if (stricmp(fontname, fam) == 0) {
		    /* Found... */
		    if (curFontPtr > lastFontPtr)
			memcpy(lastFontPtr, curFontPtr, sizeof(*curFontPtr));
		    lastFontPtr++;
		}
	      next_font:
		curFontPtr++;
	    }
	    *nptr = lastFontPtr - os2fonts;
	    if (*nptr)
		return os2fonts;
	}
	*nptr = 0;
	ckfree((char*)os2fonts);
	return NULL;
}

const char * const weightclasses[] = {
    "ERROR",
    "Ultra-light",
    "Extra-light",
    "Light",
    "Semi-light",
    "Medium (normal)",
    "Semi-bold",
    "Bold",
    "Extra-bold",
    "Ultra-bold",
};


void
dumpFontList(PFONTMETRICS curFont, int n)
{
    PFONTMETRICS lim = curFont + n;
    
    while (curFont < lim) {
	int weightClass = curFont->usWeightClass;

	if (weightClass <= 0 || weightClass >= 8)
	    weightClass = 0;
	printf("face[%s]%s, fam[%s]%s\n\tlMatch=%d, Em %d (nom %ddpt, lMBE %d), xR %d, yR %d, avWid=%d,\n",
	       curFont->szFacename,
	       (curFont->fsType & FM_TYPE_FACETRUNC) ? " (trunc)" : "",
	       curFont->szFamilyname,
	       (curFont->fsType & FM_TYPE_FAMTRUNC) ? " (trunc)" : "",
	       (int)curFont->lMatch, (int)curFont->lEmHeight,
	       (int)curFont->sNominalPointSize,
	       (int)curFont->lMaxBaselineExt, (int)curFont->sXDeviceRes,
	       (int)curFont->sYDeviceRes, (int)curFont->lAveCharWidth);
	printf("\t%s, %s, %s,\tSelection=%s%s%s%s%s%s%s\tweightClass=%s.\n",
	       (curFont->fsType & FM_TYPE_FIXED) ? "fixedW" : "proprtnl",
	       (curFont->fsType & FM_DEFN_GENERIC) ? "GPI" : "device",
	       (curFont->fsDefn & FM_DEFN_OUTLINE) ? "outline" : "bmp",
	       ((curFont->fsSelection & FM_SEL_ITALIC) ? "italic, " : ""),
	       ((curFont->fsSelection & FM_SEL_BOLD) ? "bold, " : ""),
	       ((curFont->fsSelection & FM_SEL_UNDERSCORE) ? "underlined, " : ""),
	       ((curFont->fsSelection & FM_SEL_OUTLINE) ? "outline, " : ""),
	       ((curFont->fsSelection & FM_SEL_STRIKEOUT) ? "strickeout, " : ""),
	       ((curFont->fsSelection & FM_SEL_ISO9241_TESTED) ? "iso9241-tested, " : ""),
	       ((curFont->fsSelection & FM_SEL_NEGATIVE) ? "color=negative, " : ""),
	       weightclasses[weightClass]);
	curFont++;
    }
}

/* Some fonts do not have enough info in the FONTMETRICS structure,
   relying on BOLD/ITALIC in the name...   Need to scan the name too? */
int
fontIsItalic(PFONTMETRICS fntPtr)
{
    return (fntPtr->fsSelection & FM_SEL_ITALIC);
}

int
fontIsBold(PFONTMETRICS fntPtr)
{
    return (fntPtr->fsSelection & FM_SEL_BOLD)
	|| (fntPtr->usWeightClass > 5);
}

#define WANT_ITALIC		4
#define WANT_BOLD		8

int
boldItalicError(PFONTMETRICS curFont, int flags, int *simulate)
{
    int simul = 0;
    int cerror = 0;

    if ((!!fontIsBold(curFont)) ^ (!!(flags & WANT_BOLD))) {/* xor */
	if (flags & WANT_BOLD) {
	    cerror += 10;	/* Prefer a different size */
	} else {
	    cerror += 4;	/* Can simulate, not a big deal. */
	    simul |= FATTR_SEL_ITALIC;
	}
    }
	    
    if ((!!fontIsItalic(curFont)) ^ (!!(flags & WANT_ITALIC))) {
	if (flags & WANT_ITALIC) {
	    cerror += 10;	/* Prefer a different size */
	} else {
	    cerror += 4;	/* Can simulate, not a big deal. */
	    simul |= FATTR_SEL_BOLD;
	}
    }
    *simulate = simul;
    return cerror;
}

#define HEIGHT_AS_NOMINAL	0
#define HEIGHT_AS_BASELINE	1
#define HEIGHT_AS_EM_HEIGHT	2
/* Out of the list of fonts, find one with the best match for sizes
   and the resolution.  Ignore the boldness and italicness info, let
   PM handle it later (XXXX should this be changed)?
   There are 3 different ways to choose by size. */
PFONTMETRICS
findBestFont(PFONTMETRICS fntPtr, int n, int pixels, int width, int flags, int *simulate)
{
    BOOL found = FALSE;
    LONG outline = -1;
    LONG font = 0;
    int i, bestError = INT_MAX, best = -1, retry = 0, simul, best_simul;
    PFONTMETRICS bestBitmap = NULL, lastOutline = NULL, curFont = fntPtr;
    PFONTMETRICS bestOutline = NULL;
    int bestOutlineError = INT_MAX, bestOutlineSimul;

    *simulate = 0;
    /* Examine the sizes, find the best one, preferring a bitmap font over
     * a scalable (outline) one if it exists. */
    while ((curFont < fntPtr + n) && !found) {	    
	/* Note: scalable fonts seem to return lEmHeight = 16, so first
	 * check for outline, then "point size" to not match on size 16. */
#ifdef DEBUG
	printf("    trying %s font %s (%ddp, lMaxBaselineExt %d), match %d\n",
	       (curFont->fsDefn & FM_DEFN_OUTLINE) ? "outline" : "fixed",
	       curFont->szFacename, curFont->sNominalPointSize,
	       curFont->lMaxBaselineExt, curFont->lMatch);
#endif

	if (curFont->fsDefn & FM_DEFN_OUTLINE) {
	    /* Remember we found an outline font */
	    int cerror = boldItalicError(curFont, flags, &simul);

	    if (cerror < bestOutlineError) {
		bestOutline = curFont;
		bestOutlineError = cerror;
		bestOutlineSimul = simul;
	    }
	    lastOutline = curFont;

#ifdef DEBUG
	    printf("    found outline font %s, match %d\n",
		   curFont->szFacename, curFont->lMatch);
#endif
	} else {
	    /* Bitmap font, check size, type, resolution */
	    int cerror = 0, err1, h;

	    /*
	     * Note: FONTMETRICS.fsSelection can contain FM_SEL_ISO9241_TESTED,
	     * FATTRS.fsSelection cannot.
	     */
#ifdef DEBUG
	    printf("m%d, Em %d (nom %ddpt, lMBE %d), xR %d, yR %d, %s, %s, face[%s]%s, fam[%s]%s\n",
		   curFont->lMatch, curFont->lEmHeight, curFont->sNominalPointSize,
		   curFont->lMaxBaselineExt, curFont->sXDeviceRes,
		   curFont->sYDeviceRes,
		   (curFont->fsType & FM_TYPE_FIXED) ? "fix" : "prop",
		   (curFont->fsDefn & FM_DEFN_OUTLINE) ? "outl" : "bmp",
		   curFont->szFacename,
		   (curFont->fsType & FM_TYPE_FACETRUNC) ? " (trunc()" : "",
		   curFont->szFamilyname,
		   (curFont->fsType & FM_TYPE_FAMTRUNC) ? " (trunc()" : "");
	    printf("flags %d, os2f.sNom %d, os2f.lMBE %d\n",
		   flags, curFont->sNominalPointSize,
		   curFont->lMaxBaselineExt * 10);
#endif
	    /* TCL uses HEIGHT_AS_EM_HEIGHT, it gives results which are
	       not compatible with how Presentation parameters work. */ 
	    h = (flags == HEIGHT_AS_EM_HEIGHT
		 ? (curFont->lEmHeight * 10)
		 : (flags == HEIGHT_AS_NOMINAL
		    ? curFont->sNominalPointSize
		    : (curFont->lMaxBaselineExt * 10)));
	    err1 = h - 10 * pixels;
	    if (err1 < 0)
		err1 = -err1;
	    cerror = err1;
	    if (width) {
		err1 = width - curFont->lAveCharWidth;
		if (err1 < 0)
		    err1 = -err1;
		cerror += err1 * 3;	/* 10/3 times cheaper */
	    }
#ifndef SEPARATE
	    if (curFont->sXDeviceRes != aDevCaps[CAPS_HORIZONTAL_FONT_RES]
		|| curFont->sYDeviceRes != aDevCaps[CAPS_VERTICAL_FONT_RES])
		cerror += 1;
#endif	/* SEPARATE */ 

	    cerror += boldItalicError(curFont, flags, &simul);

	    if (cerror == 0)
		return curFont;		/* Prefer an exact bitmap font. */
	    if (cerror < bestError) {
		bestError = cerror;
		bestBitmap = curFont;
		best_simul = simul;
	    }
#ifdef DEBUG
	    if (found) printf("    found bitmap font %s, match %d (size %d)\n",
			      curFont->szFacename, curFont->lMatch,
			      curFont->sNominalPointSize);
	    if (curFont->lEmHeight != pixels) {
		printf("    height %d doesn't match required %d\n",
		       curFont->lEmHeight,
		       pixels);
#ifndef SEPARATE
	    } else if (curFont->sXDeviceRes !=
		       aDevCaps[CAPS_HORIZONTAL_FONT_RES]) {
		printf("    hor. device res %d doesn't match required %d\n",
		       curFont->sXDeviceRes,
		       aDevCaps[CAPS_HORIZONTAL_FONT_RES]);
	    } else if (curFont->sYDeviceRes !=
		       aDevCaps[CAPS_VERTICAL_FONT_RES]) {
		printf("    vert. device res %d doesn't match required %d\n",
		       curFont->sYDeviceRes,
		       aDevCaps[CAPS_VERTICAL_FONT_RES]);
	    } else if ( logfonts[lFontID].fattrs.fsFontUse
			& FATTR_FONTUSE_TRANSFORMABLE) {
		printf("    transformations require outline font\n");
#endif
	    }
#endif
	}
	curFont++;
    }
    /* Prefer an approximate bitmap only if it differs only in a resolution */
    if (bestOutline && (bestError > bestOutlineError + 1)) {
	*simulate = bestOutlineSimul;
	return bestOutline;
    }
    *simulate = simul;
    return bestBitmap;
}



#define OS2_FONT_FLAGS_IN_FA	(1<<16)
#define OS2_FONT_HEIGH_NOMINAL	(2<<16)	/* Use the same height as the
					   font selection dialog */
#define OS2_FONT_BY_FAMILY	(4<<16)
LONG
OS2GetFontFromAttributes(faPtr, flags, width)
    CONST TkFontAttributes *faPtr;  /* Set of attributes to match. */
    unsigned long flags;	/* Either FATTR_*, or OS2_FONT_FLAGS_IN_FA. */
    int width;			/* Requested width in pixels, or zero. */
{
    int pixelsize, match;
    CONST char *family;
    double d;
    int oblique = 0;
    int deciPointSize, simulate = 0;
    PFONTMETRICS os2fonts, fntPtr = NULL;
    int nfonts;
    LONG lFontID;
    TkOS2Font *lFontPtr;
    int outline = 0;
    int lMatch = 0;			/* lMatch is a unique-per-device ID */
    char shortName[FACESIZE];
    USHORT idRegistry;

    if (nextLogicalFont >= MAX_NUM_LFONTS)
	return -1;

    family = faPtr->family;

    if (family == NULL)
	family = "";

    /* Step 1: fetch a list of appropriate fonts: */
    os2fonts = fetchOS2fonts(family, &nfonts, flags & OS2_FONT_BY_FAMILY);
    if (!os2fonts) { /* A translation from names common on other platforms: */
	if ((strcasecmp("Times", family) == 0)
	    || (strcasecmp("New York", family) == 0))
	    family = "Times New Roman";
	else if ((strcasecmp("Courier New", family) == 0)
		 || (strcasecmp("Monaco", family) == 0))
	    family = "Courier";
	else if ((strcasecmp("Arial", family) == 0)
		 || (strcasecmp("Geneva", family) == 0))
	    family = "Helvetica";

	/* Do some other backward-compatible replacements... */
	else if ((stricmp(family, "system") == 0) ||/* OS/2 and Windows font */
		 (stricmp(family, "device") == 0)) /* Windows font */
	    family = "System VIO";
	else if ((stricmp(family, "systemmonospaced") == 0) ||
		   (stricmp(family, "ansifixed") == 0) || /* Windows font */
		   (stricmp(family, "systemfixed") == 0) ||	/* Windows font */
		   (stricmp(family, "oemfixed") == 0)) /* Windows font */
	    family = "System Monospaced";
	else if ((stricmp(family, "systemproportional") == 0) ||
		   (stricmp(family, "ansi") == 0)) /* Windows font */
	    family = "System Proportional";
	else if ((stricmp(family, "symbol") == 0)) /* Univerally recognized font */
	    family = "Symbol Set";
	else
#if 0				/* We try GpiCreateLogFont method instead */
	    family = "";
#else
	    goto step2;
#endif
	os2fonts = fetchOS2fonts(family, &nfonts, flags & OS2_FONT_BY_FAMILY); /* Retry */

#if 0
	if (!os2fonts && *family) {	/* Should not happen? */
	    family = "";
	    os2fonts = fetchOS2fonts(family, &nfonts, flags & OS2_FONT_BY_FAMILY);
	}
#endif
    }

    /* Step 2: find the font with best matching sizes. */
  step2:
    pixelsize = -faPtr->pointsize;	/* Negative values: size in points. */
    if (pixelsize == 0)
	pixelsize = 12;
    if (pixelsize < 0) {		/* Actually in points */
	d = -pixelsize * 1. * aDevCaps[CAPS_VERTICAL_FONT_RES] / 72;
#if 0
        deciPointSize = -pixelsize*10;
        d = -pixelsize * 25.4 / 72;
	(aDevCaps[CAPS_VERTICAL_FONT_RES] * pointSize + 360) / 720
	d *= WidthOfScreen(Tk_Screen(tkwin));
	d /= WidthMMOfScreen(Tk_Screen(tkwin));
#endif
	d += 0.5;
        pixelsize = (int) d;
    }

    if (os2fonts)
	fntPtr = findBestFont(os2fonts, nfonts, pixelsize, width,
			      (((flags & OS2_FONT_HEIGH_NOMINAL) 
			       ? HEIGHT_AS_NOMINAL
			       : HEIGHT_AS_BASELINE)
			       | ((faPtr->weight > TK_FW_NORMAL)
				  ? WANT_BOLD : 0)
			       | ((faPtr->slant == TK_FS_ITALIC)
				  ? WANT_ITALIC : 0)),
			      &simulate);

    /* Step 3: extract all the info we need from this font. */
    /* All the previous work for that... */
    if (fntPtr) {
	lMatch = fntPtr->lMatch;
	strncpy(shortName, fntPtr->szFacename, FACESIZE - 1);
	/* Documented: only match and facename are needed.  Be extra safe... */
	outline = fntPtr->fsDefn & FM_DEFN_OUTLINE;
	width = fntPtr->lAveCharWidth;
	idRegistry = fntPtr->idRegistry;
	if (!outline)			/* Use new info instead */
	    pixelsize = fntPtr->lMaxBaselineExt;
    }
    if (os2fonts)
	ckfree((char *)os2fonts);

    d = (pixelsize * 720.) / aDevCaps[CAPS_VERTICAL_FONT_RES];
#if 0
    d = pixelsize / 25.4 * 720;
    d /= WidthOfScreen(Tk_Screen(tkwin));
    d *= WidthMMOfScreen(Tk_Screen(tkwin));
#endif
    d += 0.5;
    deciPointSize = (int) d;

#if 0
    if (!fntPtr)
	return -1;			/* XXXX maybe try GpiCreateLogFont? */
#endif

    /* Step 4: assign a logical id to the font. */
    if (flags & OS2_FONT_FLAGS_IN_FA) {
	flags = 0;
#if 0					/* Should be done by simulate... */
	if (faPtr->weight > TK_FW_NORMAL)
	    flags |= FATTR_SEL_BOLD;
	if (faPtr->slant == TK_FS_ITALIC)
	    flags |= FATTR_SEL_ITALIC;
#endif
	if (faPtr->slant == TK_FS_OBLIQUE)
	    flags |= FATTR_SEL_ITALIC;
	if (faPtr->underline)
	    flags |= FATTR_SEL_UNDERSCORE;
	if (faPtr->overstrike)
	    flags |= FATTR_SEL_STRIKEOUT;
    }

    /* Step 5: Set defaults in logfont */
    lFontID = nextLogicalFont;
    lFontPtr = logfonts + lFontID;
    lFontPtr->fattrs.usRecordLength = (USHORT)sizeof(FATTRS);
    lFontPtr->fattrs.fsSelection = (USHORT)((flags|simulate) & 0xFFFF);
    lFontPtr->fattrs.idRegistry = 0;	/* Unknown */
    lFontPtr->fattrs.usCodePage = 0;	/* Use present codepage */
    lFontPtr->fattrs.lMaxBaselineExt = 0L; /* 0 for vector fonts */
    lFontPtr->fattrs.lAveCharWidth = 0L; /* 0 for vector fonts */
    lFontPtr->fattrs.fsType = 0;
    lFontPtr->fattrs.fsFontUse = 0;
    if (oblique) {			/* Set to 15 degree slant forward */
	lFontPtr->shear.x = 2588*oblique; /* 10000*cos(75) */
	lFontPtr->shear.y = 9659;	/* 10000*sin(75) */
    } else {
	lFontPtr->shear.x = 0;
	lFontPtr->shear.y = 1; /* Upright characters by default */
    }
    /* Not necessary to set shear by default */
    lFontPtr->setShear = (oblique ? TRUE : FALSE);
    lFontPtr->outline = outline;

    lFontPtr->deciPoints = pixelsize * 10;

    /* Fill in the exact font metrics if we found a font */
    /* Without lMatch we cannot process fontnames which are too long? */
    memset(lFontPtr->fattrs.szFacename, '\0', FACESIZE);
    strcpy(lFontPtr->fattrs.szFacename, shortName);

    lFontPtr->fattrs.idRegistry = idRegistry;
    lFontPtr->fattrs.lMatch = lMatch;
#ifdef DEBUG
    printf("    using match %d (%s)\n", lMatch,
	   lFontPtr->fattrs.szFacename);
#endif
    if (outline) {

	if (flags & OS2_FONT_HEIGH_NOMINAL) {
	    /*  pixelsize was the number from "10.Courier" notation.
		It looks like Presentation Parameters (thus us too) consider
		this as a request for deciPointNominal/10.  Thus it is
		in fact given in points. */
	    deciPointSize = pixelsize * 10;
	    d = deciPointSize / 720. * aDevCaps[CAPS_VERTICAL_FONT_RES];
	    pixelsize = (int)(d + 0.5);

	    /*	It looks as for fonts which exist both in BMP and outline
		format on BMP fonts
		  emHeight = deciPointNominal/720 * GPI_pixel_per_inch;
		(rounded down).  Since pixelsize plays the role of
		deciPointNominal/10, and later
		GpiSetCharSet() scales emHeight to
		the specified value, we need to use a similar formula to get
		consistent sizes of BMP and outline fonts.
	    */
	}
	lFontPtr->deciPoints = pixelsize*10; /* Use for scaling */
	lFontPtr->fattrs.lAveCharWidth = 0;
	lFontPtr->fattrs.lMaxBaselineExt = 0;
	lFontPtr->fattrs.fsFontUse |= FATTR_FONTUSE_OUTLINE;
    } else {
	lFontPtr->deciPoints = deciPointSize; /* Not used? */
	lFontPtr->fattrs.lAveCharWidth = width;
	lFontPtr->fattrs.lMaxBaselineExt = pixelsize;
    }

#ifdef DEBUG
    printf("    length %d, sel %x, reg %d, cp %d, mbe %d, acw %d, type %x, fu %x\n",
	   lFontPtr->fattrs.usRecordLength,
	   lFontPtr->fattrs.fsSelection,
	   lFontPtr->fattrs.idRegistry,
	   lFontPtr->fattrs.usCodePage,
	   deciPointSize,
	   lFontPtr->fattrs.lAveCharWidth,
	   lFontPtr->fattrs.fsType, lFontPtr->fattrs.fsFontUse);
#endif

    /* Step 6: Create a logical font */
    match = GpiCreateLogFont(globalPS, NULL, lFontID,
			     &(lFontPtr->fattrs));
    if (match == GPI_ERROR) {
	/* Select default font by making facename empty */
#ifdef DEBUG
	printf("    GpiCreateLogFont ERROR %x\n", WinGetLastError(hab));
	printf("XLoadFont trying default font\n");
#endif
	memset(lFontPtr->fattrs.szFacename, '\0', FACESIZE);
	match = GpiCreateLogFont(globalPS, NULL, lFontID,
				 &(lFontPtr->fattrs));
    }
    if (match == GPI_ERROR)
	return -1;

    /* Step 7: Scale the font to the right size if outline font */
    if (lFontPtr->outline) {
	ULONG rc;
	SIZEF    sizfxBox;  /*  Character-box size in world coordinates. */

	sizfxBox.cy = MAKEFIXED(pixelsize, 0);
	sizfxBox.cx = MAKEFIXED((width ? (10*width) : pixelsize),0);
	rc = GpiSetCharBox(globalPS, &sizfxBox);
 
	if (rc == GPI_ERROR) {
	    printf("GpiSetCharBox %dx%d ERROR %#x\n", 10*width,
		   lFontPtr->deciPoints, WinGetLastError(hab));
	    exit(1);
	}
    }

    rc = GpiSetCharSet(globalPS, lFontID);
 
    if (rc == GPI_ERROR) {
	printf("GpiSetCharSet ERROR %#x\n", WinGetLastError(hab));
	exit(1);
    }
    if (lFontPtr->outline) {
#if 0
	rc = GpiQueryCharBox(globalPS, &charBox);
	if (rc!=TRUE) {
	    printf("GpiQueryCharBox ERROR %x\n");
	} else {
	    printf("GpiQueryCharBox OK: now cx %d (%d,%d), cy %d (%d,%d)\n",
		   charBox.cx, FIXEDINT(charBox.cx), FIXEDFRAC(charBox.cx),
		   charBox.cy, FIXEDINT(charBox.cy), FIXEDFRAC(charBox.cy));
	}
#endif
    }

#ifdef DEBUG
    printf("XLoadFont match %s lFontID %d\n", match==FONT_MATCH ?
	   "FONT_MATCH" : (match==FONT_DEFAULT ? "FONT_DEFAULT" :
			   "GPI_ERROR"), lFontID);
#endif

    if (1) {
	ULONG rc;
	FONTMETRICS fm;

	rc = GpiQueryFontMetrics(globalPS, sizeof(FONTMETRICS), &fm);
	if (rc != TRUE) {
	    printf("GpiQueryFontMetrics ERROR %#x\n", WinGetLastError(hab));
	    exit(1);
	}
	dumpFontList(&fm,1);
    }
    nextLogicalFont++;
    return lFontID;
}

/*
 *---------------------------------------------------------------------------
 *
 * TkpGetNativeFont --
 *
 *	Map a platform-specific native font name to a TkFont.
 *
 * Results:
 * 	The return value is a pointer to a TkFont that represents the
 *	native font.  If a native font by the given name could not be
 *	found, the return value is NULL.  
 *
 *	Every call to this procedure returns a new TkFont structure,
 *	even if the name has already been seen before.  The caller should
 *	call TkpDeleteFont() when the font is no longer needed.
 *
 *	The caller is responsible for initializing the memory associated
 *	with the generic TkFont when this function returns and releasing
 *	the contents of the generic TkFont before calling TkpDeleteFont().
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

TkFont *
TkpGetNativeFont(tkwin, name)
    Tk_Window tkwin;		/* For display where font will be used. */
    CONST char *name;		/* Platform-specific font name. */
{
    XFontStruct *fontStructPtr;
    TkFontAttributes fa;
    char *fontname = NULL;
    int pixelsize = 0;
    char buff[180];
    int l, off = 0, retval;
    CONST char *ininame, *ininame_end = 0;
    CONST char *try_fontname = 0, *try_fontname_end = 0;
    unsigned long flags = 0;

    /* Recognize names which are of "OS2 Presentation Parameters" (generalized)
       form:  Pixelsize.[attr.]*Fontname[.attr]*  */

    if (!(sscanf(name, "%d.%n", &pixelsize, &off) && off > 0))
	return NULL;

    /*
     * The following code suggested by Ilya Zakharevich.
     * Its use is to allow font selection "in OS/2-style", like
     * "10.Courier".
     * Ilya's way of supplying attributes of the font is against
     * the documented "pointSize.Fontname[.attr ...]" though,
     * because it gives attributes between the pointsize and the
     * name of the font.
     * [Tcl takek the "official" stance and also supply the rest of the
     * font Presentation Parameters: underline, strikeout, outline.]
     * We do not.
     */

#ifdef DEBUG
    printf("    trying Presentation Parameters-notation font: d %d, n %d\n", l, off);
#endif
    name += off;
    ininame = name;
    /* Split by '.' into words, the fontname name is either the only
       component which is not in the predefined list, or. if none such,
       the first component.  */
    while (*name) {
	char *cmp;
	char buf[180];
	int nfields;
	unsigned long flag;

	nfields = sscanf(name, "%179[^.]%n", buf, &off);
#ifdef DEBUG
	printf("    sscanf returns %d, off %d\n", nfields, off);
#endif
	if (nfields < 1)
	    return NULL;
	switch (off) {
	case 4:
	    cmp = "bold"; 
	    flag = FATTR_SEL_BOLD;
	    break;
	case 6:
	    cmp = "italic"; 
	    flag = FATTR_SEL_ITALIC;
	    break;
	case 7:
	    cmp = "outline";
	    flag = FATTR_SEL_OUTLINE;
	    break;
	case 9:
	    cmp = (*name == 's' ? "strikeout" : "underline");
	    flag = (*name == 's' ? FATTR_SEL_STRIKEOUT 
		    : FATTR_SEL_UNDERSCORE);
	    break;
	default:
	    cmp = NULL;
	}
	if (cmp && strnicmp(name, cmp, off) == 0)
	    flags |= flag;
	else if (try_fontname)		/* Already got a candidate */
	    return NULL;
	else
	    try_fontname = name;
	name += off;
	if (*name != '.' && *name)
	    return NULL;
	if (!ininame_end)
	    ininame_end = name;
	if (try_fontname && !try_fontname_end)
	    try_fontname_end = name;
	if (*name)
	    name++;
    }

    if (!fontname) {
	if (!try_fontname) {
	    try_fontname = ininame;
	    try_fontname_end = ininame_end;
	}
	strncpy(buff,try_fontname, try_fontname_end - try_fontname + 1);
	buff[try_fontname_end - try_fontname] = 0;
    }

    memset(&fa, 0, sizeof(fa));
    fa.family = buff;
    fa.pointsize = -pixelsize;
    /* Use the same notion of height as the font selection dialogue. */
    retval = OS2GetFontFromAttributes(&fa, flags | OS2_FONT_HEIGH_NOMINAL, 0);
    if (retval == -1)
	return NULL;
    fontStructPtr = OS2QueryFont(retval);
    if (!fontStructPtr)
	return NULL;
    return (TkFont *) AllocFont(NULL, tkwin, fontStructPtr, fa.family);
}


/*
 *---------------------------------------------------------------------------
 *
 * TkpGetFontFromAttributes -- 
 *
 *	Given a desired set of attributes for a font, find a font with
 *	the closest matching attributes.
 *
 * Results:
 * 	The return value is a pointer to a TkFont that represents the
 *	font with the desired attributes.  If a font with the desired
 *	attributes could not be constructed, some other font will be
 *	substituted automatically.
 *
 *	Every call to this procedure returns a new TkFont structure,
 *	even if the specified attributes have already been seen before.
 *	The caller should call TkpDeleteFont() to free the platform-
 *	specific data when the font is no longer needed.  
 *
 *	The caller is responsible for initializing the memory associated
 *	with the generic TkFont when this function returns and releasing
 *	the contents of the generic TkFont before calling TkpDeleteFont().
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
TkFont *
TkpGetFontFromAttributes(tkFontPtr, tkwin, faPtr)
    TkFont *tkFontPtr;		/* If non-NULL, store the information in
				 * this existing TkFont structure, rather than
				 * allocating a new structure to hold the
				 * font; the existing contents of the font
				 * will be released.  If NULL, a new TkFont
				 * structure is allocated. */
    Tk_Window tkwin;		/* For display where font will be used. */
    CONST TkFontAttributes *faPtr;  /* Set of attributes to match. */
{
    int numNames, score, i, scaleable, pixelsize, xaPixelsize;
    int bestIdx, bestScore, bestScaleableIdx, bestScaleableScore;
    TkXLFDAttributes xa;    
    char buf[256];
    OS2Font *fontPtr;
    char **nameList;
    XFontStruct *fontStructPtr;
    CONST char *fmt, *family;
    double d;
    LONG retval;

    family = faPtr->family;
    if (family == NULL) {
	family = "*";
    }

    pixelsize = -faPtr->pointsize;
    if (pixelsize < 0) {
        d = -pixelsize * 25.4 / 72;
	d *= WidthOfScreen(Tk_Screen(tkwin));
	d /= WidthMMOfScreen(Tk_Screen(tkwin));
	d += 0.5;
        pixelsize = (int) d;
    }

    retval = OS2GetFontFromAttributes(faPtr, OS2_FONT_FLAGS_IN_FA | OS2_FONT_BY_FAMILY, 1);
    if (retval == -1)
	return NULL;
    fontStructPtr = OS2QueryFont(retval);
    if (!fontStructPtr)
	return NULL;
    return (TkFont *) AllocFont(NULL, tkwin, fontStructPtr, faPtr->family);
}

/*
 *----------------------------------------------------------------------
 *
 * XQueryFont --
 *
 *	Retrieve information about the specified font.
 *
 * Results:
 *	Returns a newly allocated XFontStruct.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

XFontStruct *
OS2QueryFont(font_ID)
    LONG font_ID;
{
    XFontStruct *fontPtr = (XFontStruct *) ckalloc(sizeof(XFontStruct));
    LONG oldFont;
    FONTMETRICS fm;
    XCharStruct bounds;
    BOOL rc;
    POINTL noShear= {0, 1};

#ifdef DEBUG
printf("XQueryFont FID %d\n", font_ID);
#endif

    if (!fontPtr) {
	return NULL;
    }
    
    fontPtr->fid = font_ID;

    oldFont = GpiQueryCharSet(globalPS);
    rc = GpiSetCharSet(globalPS, (LONG) fontPtr->fid);
#ifdef DEBUG
printf("GpiSetCharSet %d returns %d\n", fontPtr->fid, rc);
if (rc==FALSE) printf("Error=%x\n", WinGetLastError(hab));
#endif
    /* Set slant if necessary */
    if (logfonts[(LONG)fontPtr->fid].setShear) {
        GpiSetCharShear(globalPS, &(logfonts[(LONG)fontPtr->fid].shear));
    }

    /*
     * Determine the font metrics and store the values into the appropriate
     * X data structures.
     */

    /* If this is an outline font, set the char box */
    if (logfonts[font_ID].outline) {
#ifdef DEBUG
        SIZEF charBox;
#endif
        rc = TkOS2ScaleFont(globalPS, logfonts[font_ID].deciPoints, 0);
#ifdef DEBUG
        if (rc!=TRUE) {
            printf("TkOS2ScaleFont %d ERROR %x\n", logfonts[font_ID].deciPoints,
                   WinGetLastError(hab));
        } else {
            printf("TkOS2ScaleFont %d OK\n", logfonts[font_ID].deciPoints);
        }
        rc = GpiQueryCharBox(globalPS, &charBox);
        if (rc!=TRUE) {
            printf("GpiQueryCharBox ERROR %x\n");
        } else {
            printf("GpiQueryCharBox OK: now cx %d (%d,%d), cy %d (%d,%d)\n",
                   charBox.cx, FIXEDINT(charBox.cx), FIXEDFRAC(charBox.cx),
                   charBox.cy, FIXEDINT(charBox.cy), FIXEDFRAC(charBox.cy));
        }
#endif
    }

    if (GpiQueryFontMetrics(globalPS, sizeof(FONTMETRICS), &fm)) {
#ifdef DEBUG
        printf("Font metrics: first %c (%d), last %c (%d), def %c (%d),
               maxCharInc %d, maxAsc %d, maxDesc %d, maxBaselineExt %d,
               bounds %d\n", fm.sFirstChar, fm.sFirstChar, fm.sLastChar,
               fm.sLastChar, fm.sDefaultChar, fm.sDefaultChar, fm.lMaxCharInc,
               fm.lMaxAscender, fm.lMaxDescender, fm.lMaxBaselineExt, bounds);
        printf("    sNominalPointSize %d, size %d\n", fm.sNominalPointSize, 
               fm.lMaxBaselineExt);
        printf("    lEmInc %d, underpos %d, size %d\n", fm.lEmInc,
               fm.lUnderscorePosition, fm.lUnderscoreSize);
#endif

        /*
         * Copy fontmetrics into own structure, so we remember scalable font
         * modifications.
         */
        memcpy((void *)&logfonts[font_ID].fm, (void *)&fm, sizeof(FONTMETRICS));
#ifdef DEBUG
        printf("    sXDeviceRes %d, sYDeviceRes %d\n",
               logfonts[font_ID].fm.sXDeviceRes,
               logfonts[font_ID].fm.sYDeviceRes);
#endif

	fontPtr->direction = LOBYTE(fm.sInlineDir) < 90 || LOBYTE(fm.sInlineDir) > 270
	                     ? FontLeftToRight : FontRightToLeft;
	fontPtr->min_byte1 = 0;
	fontPtr->max_byte1 = 0;
	fontPtr->min_char_or_byte2 = fm.sFirstChar;
	/* sLastChar is *offset* from sFirstChar! */
	fontPtr->max_char_or_byte2 = fm.sFirstChar + fm.sLastChar;
	fontPtr->all_chars_exist = True;
	fontPtr->default_char = fm.sDefaultChar;
	fontPtr->n_properties = 0;
	fontPtr->properties = NULL;
	bounds.lbearing = 0;
	bounds.rbearing = fm.lMaxCharInc;
	bounds.width = fm.lMaxCharInc;
	bounds.ascent = fm.lMaxAscender;
	bounds.descent = fm.lMaxDescender;
	bounds.attributes = 0;
	fontPtr->ascent = fm.lMaxAscender;
	fontPtr->descent = fm.lMaxDescender;
	fontPtr->min_bounds = bounds;
	fontPtr->max_bounds = bounds;

	/*
	 * If the font is not fixed pitch, then we need to construct
	 * the per_char array.
	 */

	if ( !(fm.fsType & FM_TYPE_FIXED) ) {
	    int i;
	    char c;
            /*
             * sLastChar is offset from sFirstChar
             * Only 256 code points (0 - 255) allowed in a code page.
             */
	    int nchars = MIN(((int)fm.sLastChar + 1), 255);
	    int minWidth = 30000;
POINTL endPoint[2];
            PLONG widths;
	    fontPtr->per_char =
		(XCharStruct *)ckalloc(sizeof(XCharStruct) * nchars);

            /*
             * GpiQueryCharStringPos seems to be more precise than
             * GpiQueryWidthTable. In the widget demo, the first only exhibits
             * some "dancing" when highlighting, while the latter gives a very
             * noticeable space before the cursor.
             * On the other hand, it is abysmal for raster fonts.
             */
            if (logfonts[font_ID].outline) {
                for (i = 0, c = fm.sFirstChar; i < nchars ; i++, c++ ) {
                    GpiQueryCharStringPos(globalPS, 0L, 1, &c, NULL, endPoint);
                    /*
                    fontPtr->per_char[i] = bounds;
                    */
                    fontPtr->per_char[i].width = endPoint[1].x - endPoint[0].x;
#ifdef DEBUG
                    printf("    width [%c] %d (endP1.x %d, endP0.x %d)\n",
                           fm.sFirstChar+i, fontPtr->per_char[i].width,
                           endPoint[1].x, endPoint[0].x);
#endif
                    if (minWidth > fontPtr->per_char[i].width) {
                        minWidth = fontPtr->per_char[i].width;
                    }
                }
            } else {
                /* Bitmap -> use GpiQueryWidthTable */
                widths = (PLONG) ckalloc(sizeof(LONG)*nchars);
                if (widths != NULL) {
                    rc= GpiQueryWidthTable(globalPS, fm.sFirstChar, nchars,
                                           widths);
                    if (rc == TRUE) {
                        for (i = 0; i < nchars ; i++) {
                            fontPtr->per_char[i] = bounds;
                            fontPtr->per_char[i].width = widths[i];
#ifdef DEBUG
                            printf("    width (bmp) [%c] %d\n", fm.sFirstChar+i,
                                   fontPtr->per_char[i].width);
#endif
                            if (minWidth > fontPtr->per_char[i].width) {
                	            minWidth = fontPtr->per_char[i].width;
                            }
                        }
#ifdef DEBUG
                        printf("variable pitch, nchars %d, minWidth %d\n", nchars,
                               minWidth);
#endif
                    } else {
    	                ckfree((char *)fontPtr);
                        fontPtr = NULL;
#ifdef DEBUG
                        printf("    GpiQueryWidthTable %d ERROR %x\n", nchars,
                               WinGetLastError(hab));
#endif
                    }
                } else {
    	            ckfree((char *)fontPtr);
                    fontPtr = NULL;
#ifdef DEBUG
                    printf("    couldn't allocate memory for widths\n");
#endif
                    goto restore;
                }
            }

	    fontPtr->min_bounds.width = minWidth;
	} else {
#ifdef DEBUG
            printf("    Fixed pitch font\n");
#endif
	    fontPtr->per_char = NULL;
	}
    } else {
	ckfree((char *)fontPtr);
	fontPtr = NULL;
    }    

restore:

    /* Restore font */
    if (logfonts[(LONG)fontPtr->fid].setShear) {
        GpiSetCharShear(globalPS, &noShear);
    }
    GpiSetCharSet(globalPS, oldFont);
    
    return fontPtr;
}

/*
 *---------------------------------------------------------------------------
 *
 * TkpDeleteFont --
 *
 *	Called to release a font allocated by TkpGetNativeFont() or
 *	TkpGetFontFromAttributes().  The caller should have already
 *	released the fields of the TkFont that are used exclusively by
 *	the generic TkFont code.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	TkFont is deallocated.
 *
 *---------------------------------------------------------------------------
 */

void
TkpDeleteFont(tkFontPtr)
    TkFont *tkFontPtr;		/* Token of font to be deleted. */
{
    OS2Font *fontPtr;

    fontPtr = (OS2Font *) tkFontPtr;

    XFreeFont(fontPtr->display, fontPtr->fontStructPtr);
    ckfree((char *) fontPtr);
}

/*
 *---------------------------------------------------------------------------
 *
 * TkpGetFontFamilies --
 *
 *	Return information about the font families that are available
 *	on the display of the given window.
 *
 * Results:
 *	interp->result is modified to hold a list of all the available
 *	font families.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
 
void
TkpGetFontFamilies(interp, tkwin)
    Tcl_Interp *interp;
    Tk_Window tkwin;
{
    int i, new, numNames;
    char *family, *end, *p;
    Tcl_HashTable familyTable;
    Tcl_HashEntry *hPtr;
    Tcl_HashSearch search;
    char **nameList;
    PFONTMETRICS fonts = fetchOS2fonts(NULL, &numNames, 0);

    Tcl_InitHashTable(&familyTable, TCL_STRING_KEYS);

    for (i = 0; i < numNames; i++) {
	family = fonts[i].szFamilyname;
	for (p = family; *p != '\0'; p++) {
	    if (isupper(UCHAR(*p))) {
		*p = tolower(UCHAR(*p));
	    }
	}
	Tcl_CreateHashEntry(&familyTable, family, &new);
    }

    hPtr = Tcl_FirstHashEntry(&familyTable, &search);
    while (hPtr != NULL) {
	Tcl_AppendElement(interp, Tcl_GetHashKey(&familyTable, hPtr));
	hPtr = Tcl_NextHashEntry(&search);
    }

    Tcl_DeleteHashTable(&familyTable);
    ckfree((char*)fonts);
}

/*
 *---------------------------------------------------------------------------
 *
 *  Tk_XMeasureChars --
 *
 *	Determine the number of characters from the string that will fit
 *	in the given horizontal span.  The measurement is done under the
 *	assumption that Tk_DrawChars() will be used to actually display
 *	the characters.
 *
 *	This function approximates the result by summing up widths of chars.
 *
 * Results:
 *	The return value is the number of characters from source that
 *	fit into the span that extends from 0 to maxLength.  *lengthPtr is
 *	filled with the x-coordinate of the right edge of the last
 *	character that did fit.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
int
Tk_XMeasureChars(tkfont, source, numChars, maxLength, flags, lengthPtr)
    Tk_Font tkfont;		/* Font in which characters will be drawn. */
    CONST char *source;		/* Characters to be displayed.  Need not be
				 * '\0' terminated. */
    int numChars;		/* Maximum number of characters to consider
				 * from source string. */
    int maxLength;		/* If > 0, maxLength specifies the longest
				 * permissible line length; don't consider any
				 * character that would cross this
				 * x-position.  If <= 0, then line length is
				 * unbounded and the flags argument is
				 * ignored. */
    int flags;			/* Various flag bits OR-ed together:
				 * TK_PARTIAL_OK means include the last char
				 * which only partially fit on this line.
				 * TK_WHOLE_WORDS means stop on a word
				 * boundary, if possible.
				 * TK_AT_LEAST_ONE means return at least one
				 * character even if no characters fit. */
    int *lengthPtr;		/* Filled with x-location just after the
				 * terminating character. */
{
    OS2Font *fontPtr;
    CONST char *p;		/* Current character. */
    CONST char *term;		/* Pointer to most recent character that
				 * may legally be a terminating character. */
    int termX;			/* X-position just after term. */
    int curX;			/* X-position corresponding to p. */
    int newX;			/* X-position corresponding to p+1. */
    int c, sawNonSpace;

    fontPtr = (OS2Font *) tkfont;

    if (numChars == 0) {
	*lengthPtr = 0;
	return 0;
    }

    if (maxLength <= 0) {
	maxLength = INT_MAX;
    }

    newX = curX = termX = 0;
    p = term = source;
    sawNonSpace = !isspace(UCHAR(*p));

    /*
     * Scan the input string one character at a time, calculating width.
     */

    for (c = UCHAR(*p); ; ) {
	newX += fontPtr->widths[c];
	if (newX > maxLength) {
	    break;
	}
	curX = newX;
	numChars--;
	p++;
	if (numChars == 0) {
	    term = p;
	    termX = curX;
	    break;
	}

	c = UCHAR(*p);
	if (isspace(c)) {
	    if (sawNonSpace) {
		term = p;
		termX = curX;
		sawNonSpace = 0;
	    }
	} else {
	    sawNonSpace = 1;
	}
    }

    /*
     * P points to the first character that doesn't fit in the desired
     * span.  Use the flags to figure out what to return.
     */

    if ((flags & TK_PARTIAL_OK) && (numChars > 0) && (curX < maxLength)) {
	/*
	 * Include the first character that didn't quite fit in the desired
	 * span.  The width returned will include the width of that extra
	 * character.
	 */

	numChars--;
	curX = newX;
	p++;
    }
    if ((flags & TK_AT_LEAST_ONE) && (term == source) && (numChars > 0)) {
	term = p;
	termX = curX;
	if (term == source) {
	    term++;
	    termX = newX;
	}
    } else if ((numChars == 0) || !(flags & TK_WHOLE_WORDS)) {
	term = p;
	termX = curX;
    }

    *lengthPtr = termX;
    return term-source;
}


/*
 *---------------------------------------------------------------------------
 *
 *  Tk_MeasureChars --
 *
 *	Determine the number of characters from the string that will fit
 *	in the given horizontal span.  The measurement is done under the
 *	assumption that Tk_DrawChars() will be used to actually display
 *	the characters.
 *
 * Results:
 *	The return value is the number of characters from source that
 *	fit into the span that extends from 0 to maxLength.  *lengthPtr is
 *	filled with the x-coordinate of the right edge of the last
 *	character that did fit.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */
int
Tk_MeasureChars(tkfont, source, numChars, maxLength, flags, lengthPtr)
    Tk_Font tkfont;		/* Font in which characters will be drawn. */
    CONST char *source;		/* Characters to be displayed.  Need not be
				 * '\0' terminated. */
    int numChars;		/* Maximum number of characters to consider
				 * from source string. */
    int maxLength;		/* If > 0, maxLength specifies the longest
				 * permissible line length; don't consider any
				 * character that would cross this
				 * x-position.  If <= 0, then line length is
				 * unbounded and the flags argument is
				 * ignored. */
    int flags;			/* Various flag bits OR-ed together:
				 * TK_PARTIAL_OK means include the last char
				 * which only partially fit on this line.
				 * TK_WHOLE_WORDS means stop on a word
				 * boundary, if possible.
				 * TK_AT_LEAST_ONE means return at least one
				 * character even if no characters fit. */
    int *lengthPtr;		/* Filled with x-location just after the
				 * terminating character. */
{
    OS2Font *fontPtr;
    int curX, curIdx;
    TkOS2Font *lFontPtr;
    LONG rc;
    POINTL endPoints[200];
    POINTL *endPoint;

    if (numChars == 0) {
        *lengthPtr = 0;
	return 0;
    }

    fontPtr = (OS2Font *) tkfont;
    lFontPtr = logfonts + fontPtr->font.fid;

    if (lFontPtr->outline == 0)
	return Tk_XMeasureChars(tkfont, source, numChars, maxLength, flags, lengthPtr);

    if (numChars + 1 >= sizeof(endPoints)/sizeof(POINTL))
	endPoint = (POINTL*)ckalloc(sizeof(POINTL) * (numChars + 1));
    else
	endPoint = endPoints;

    if (!endPoint)			/* Cannot do anything reasonable... */
	return Tk_XMeasureChars(tkfont, source, numChars, maxLength, flags, lengthPtr);

    rc = GpiSetCharSet(globalPS, (LONG) fontPtr->font.fid);
    rc = TkOS2ScaleFont(globalPS, lFontPtr->deciPoints, 0);
    GpiQueryCharStringPos(globalPS, 0L, numChars, source, NULL, endPoint);

    if (maxLength <= 0) {
	curIdx = numChars;
	curX = endPoint[numChars].x - endPoint[0].x;
    } else {
	int max;
	int lim;
	
	max = 0;
	lim = endPoint[0].x + maxLength;
	while (endPoint[max].x <= lim && max < numChars)
	    max++;

	if ((flags & TK_WHOLE_WORDS) && max < numChars) {
	    int sawSpace;
	    int i;
	    
	    sawSpace = 0;
	    i = max;
	    while (i >= 0 && !isspace(source[i])) {
    		--i;
            }
	    while (i >= 0 && isspace(source[i])) {
		sawSpace = 1;
		--i;
            }

	    /*
	     * If a space char was not found, and the flag for forcing
	     * at least on (or more) chars to be drawn is false, then
	     * set MAX to zero so no text is drawn.  Otherwise, if a
	     * space was found, set max to be one char past the space.
	     */
	    
	    if ((i < 0) && !(flags & TK_AT_LEAST_ONE)) {
		max = 0;
	    } else if (sawSpace) {
		max = i + 1;
	    }
		
	}

	if (max == 0) {
	    curX = 0;
	} else {
	    curX = endPoint[max].x - endPoint[0].x;
	}

	if (((flags & TK_PARTIAL_OK) && max < numChars && curX < maxLength)
		|| ((flags & TK_AT_LEAST_ONE) && max == 0 && numChars > 0)) {
	    ++max;
	    curX = endPoint[max].x - endPoint[0].x;
	} 
		
	curIdx = max;
    }

    if (endPoint != endPoints)
	ckfree((char *) endPoint);

    *lengthPtr = curX;
    return curIdx;
}

/*
 *---------------------------------------------------------------------------
 *
 * Tk_DrawChars, DrawChars --
 *
 *	Draw a string of characters on the screen.  Tk_DrawChars()
 *	expands control characters that occur in the string to \X or
 *	\xXX sequences.  DrawChars() just draws the strings.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information gets drawn on the screen.
 *
 *---------------------------------------------------------------------------
 */

void
Tk_DrawChars(display, drawable, gc, tkfont, source, numChars, x, y)
    Display *display;		/* Display on which to draw. */
    Drawable drawable;		/* Window or pixmap in which to draw. */
    GC gc;			/* Graphics context for drawing characters. */
    Tk_Font tkfont;		/* Font in which characters will be drawn;
				 * must be the same as font used in GC. */
    CONST char *source;		/* Characters to be displayed.  Need not be
				 * '\0' terminated.  All Tk meta-characters
				 * (tabs, control characters, and newlines)
				 * should be stripped out of the string that
				 * is passed to this function.  If they are
				 * not stripped out, they will be displayed as
				 * regular printing characters. */
    int numChars;		/* Number of characters in string. */
    int x, y;			/* Coordinates at which to place origin of
				 * string when drawing. */
{
    OS2Font *fontPtr;
    CONST char *p;
    int i, type;
    char buf[4];

    fontPtr = (OS2Font *) tkfont;

    p = source;
    for (i = 0; i < numChars; i++) {
	type = fontPtr->types[UCHAR(*p)];
	if (type != NORMAL) {
	    DrawChars(display, drawable, gc, fontPtr, source, p - source, x, y);
	    x += XTextWidth(fontPtr->fontStructPtr, source, p - source);
	    if (type == REPLACE) {
		DrawChars(display, drawable, gc, fontPtr, buf,
			GetControlCharSubst(UCHAR(*p), buf), x, y);
		x += fontPtr->widths[UCHAR(*p)];
	    }
	    source = p + 1;
	}
	p++;
    }

    DrawChars(display, drawable, gc, fontPtr, source, p - source, x, y);
}

static void
DrawChars(display, drawable, gc, fontPtr, source, numChars, x, y)
    Display *display;		/* Display on which to draw. */
    Drawable drawable;		/* Window or pixmap in which to draw. */
    GC gc;			/* Graphics context for drawing characters. */
    OS2Font *fontPtr;		/* Font in which characters will be drawn;
				 * must be the same as font used in GC. */
    CONST char *source;		/* Characters to be displayed.  Need not be
				 * '\0' terminated.  All Tk meta-characters
				 * (tabs, control characters, and newlines)
				 * should be stripped out of the string that
				 * is passed to this function.  If they are
				 * not stripped out, they will be displayed as
				 * regular printing characters. */
    int numChars;		/* Number of characters in string. */
    int x, y;			/* Coordinates at which to place origin of
				 * string when drawing. */
{
    /*
     * Perform a quick sanity check to ensure we won't overflow the X
     * coordinate space.
     */
    
    if ((x + (fontPtr->fontStructPtr->max_bounds.width * numChars) > 0x7fff)) {
	int length;

	/*
	 * The string we are being asked to draw is too big and would overflow
	 * the X coordinate space.  Unfortunatley X servers aren't too bright
	 * and so they won't deal with this case cleanly.  We need to truncate
	 * the string before sending it to X.
	 */

	numChars = Tk_MeasureChars((Tk_Font) fontPtr, source, numChars,
		0x7fff - x, 0, &length);
    }

    XDrawString(display, drawable, gc, x, y, source, numChars);

    if (fontPtr->font.fa.underline != 0) {
	XFillRectangle(display, drawable, gc, x,
		y + fontPtr->underlinePos,
		(unsigned) XTextWidth(fontPtr->fontStructPtr, source, numChars),
		(unsigned) fontPtr->barHeight);
    }
    if (fontPtr->font.fa.overstrike != 0) {
	y -= fontPtr->font.fm.descent + (fontPtr->font.fm.ascent) / 10;
	XFillRectangle(display, drawable, gc, x, y,
		(unsigned) XTextWidth(fontPtr->fontStructPtr, source, numChars),
		(unsigned) fontPtr->barHeight);
    }
}

/*
 *---------------------------------------------------------------------------
 *
 * AllocFont --
 *
 *	Helper for TkpGetNativeFont() and TkpGetFontFromAttributes().
 *	Allocates and intializes the memory for a new TkFont that
 *	wraps the platform-specific data.
 *
 * Results:
 *	Returns pointer to newly constructed TkFont.  
 *
 *	The caller is responsible for initializing the fields of the
 *	TkFont that are used exclusively by the generic TkFont code, and
 *	for releasing those fields before calling TkpDeleteFont().
 *
 * Side effects:
 *	Memory allocated.
 *
 *---------------------------------------------------------------------------
 */ 

static OS2Font *
AllocFont(tkFontPtr, tkwin, fontStructPtr, fontName)
    TkFont *tkFontPtr;		/* If non-NULL, store the information in
				 * this existing TkFont structure, rather than
				 * allocating a new structure to hold the
				 * font; the existing contents of the font
				 * will be released.  If NULL, a new TkFont
				 * structure is allocated. */
    Tk_Window tkwin;		/* For display where font will be used. */
    XFontStruct *fontStructPtr;	/* X information about font. */
    CONST char *fontName;	/* The string passed to XLoadQueryFont() to
				 * construct the fontStructPtr. */
{
    OS2Font *fontPtr;
    unsigned long value;
    int i, width, firstChar, lastChar, n, replaceOK;
    char *name, *p;
    char buf[4];
    TkXLFDAttributes xa;
    double d;
    
    if (tkFontPtr != NULL) {
	fontPtr = (OS2Font *) tkFontPtr;
	XFreeFont(fontPtr->display, fontPtr->fontStructPtr);
    } else {
        fontPtr = (OS2Font *) ckalloc(sizeof(OS2Font));
    }

    /*
     * Encapsulate the generic stuff in the TkFont. 
     */

    fontPtr->font.fid		= fontStructPtr->fid;

    if (XGetFontProperty(fontStructPtr, XA_FONT, &value) && (value != 0)) {
	name = Tk_GetAtomName(tkwin, (Atom) value);
	TkInitFontAttributes(&xa.fa);
	if (TkParseXLFD(name, &xa) == TCL_OK) {
	    goto ok;
	}
    }
    TkInitFontAttributes(&xa.fa);
    if (TkParseXLFD(fontName, &xa) != TCL_OK) {
	TkInitFontAttributes(&fontPtr->font.fa);
	fontPtr->font.fa.family = Tk_GetUid(fontName);
    } else {
	ok:
	fontPtr->font.fa = xa.fa;
    }

    if (fontPtr->font.fa.pointsize < 0) {
	d = -fontPtr->font.fa.pointsize * 72 / 25.4;
	d *= WidthMMOfScreen(Tk_Screen(tkwin));
	d /= WidthOfScreen(Tk_Screen(tkwin));
	d += 0.5;
	fontPtr->font.fa.pointsize = (int) d;
    }
	
    fontPtr->font.fm.ascent	= fontStructPtr->ascent;
    fontPtr->font.fm.descent	= fontStructPtr->descent;
    fontPtr->font.fm.maxWidth	= fontStructPtr->max_bounds.width;
    fontPtr->font.fm.fixed	= 1;
    fontPtr->display		= Tk_Display(tkwin);
    fontPtr->fontStructPtr	= fontStructPtr;

    /*
     * Classify the characters.
     */

    firstChar = fontStructPtr->min_char_or_byte2;
    lastChar = fontStructPtr->max_char_or_byte2;
    for (i = 0; i < 256; i++) {
	if ((i == 0177) || (i < firstChar) || (i > lastChar)) {
	    fontPtr->types[i] = REPLACE;
	} else {
	    fontPtr->types[i] = NORMAL;
	}
    }

    /*
     * Compute the widths for all the normal characters.  Any other
     * characters are given an initial width of 0.  Also, this determines
     * if this is a fixed or variable width font, by comparing the widths
     * of all the normal characters.
     */

    width = 0;
    for (i = 0; i < 256; i++) {
	if (fontPtr->types[i] != NORMAL) {
	    n = 0;
	} else if (fontStructPtr->per_char == NULL) {
	    n = fontStructPtr->max_bounds.width;
	} else {
	    n = fontStructPtr->per_char[i - firstChar].width;
	}
	fontPtr->widths[i] = n;
	if (n != 0) {
	    if (width == 0) {
		width = n;
	    } else if (width != n) {
		fontPtr->font.fm.fixed = 0;
	    }
	}
    }

    /*
     * Compute the widths of the characters that should be replaced with
     * control character expansions.  If the appropriate chars are not
     * available in this font, then control character expansions will not
     * be used; control chars will be invisible & zero-width.
     */

    replaceOK = 1;
    for (p = hexChars; *p != '\0'; p++) {
	if ((UCHAR(*p) < firstChar) || (UCHAR(*p) > lastChar)) {
	    replaceOK = 0;
	    break;
	}
    }
    for (i = 0; i < 256; i++) {
	if (fontPtr->types[i] == REPLACE) {
	    if (replaceOK) {
		n = GetControlCharSubst(i, buf);
		for ( ; --n >= 0; ) {
		    fontPtr->widths[i] += fontPtr->widths[UCHAR(buf[n])];
		}
	    } else {
		fontPtr->types[i] = SKIP;
	    }
	}
    }

    if (XGetFontProperty(fontStructPtr, XA_UNDERLINE_POSITION, &value)) {
	fontPtr->underlinePos = value;
    } else {
	/*
	 * If the XA_UNDERLINE_POSITION property does not exist, the X
	 * manual recommends using the following value:
	 */

	fontPtr->underlinePos = fontStructPtr->descent / 2;
    }
    fontPtr->barHeight = 0;
    if (XGetFontProperty(fontStructPtr, XA_UNDERLINE_THICKNESS, &value)) {
	/*
	 * Sometimes this is 0 even though it shouldn't be.
	 */
	fontPtr->barHeight = value;
    }
    if (fontPtr->barHeight == 0) {
	/*
	 * If the XA_UNDERLINE_THICKNESS property does not exist, the X
	 * manual recommends using the width of the stem on a capital
	 * letter.  I don't know of a way to get the stem width of a letter,
	 * so guess and use 1/3 the width of a capital I.
	 */

	fontPtr->barHeight = fontPtr->widths['I'] / 3;
	if (fontPtr->barHeight == 0) {
	    fontPtr->barHeight = 1;
	}
    }
    if (fontPtr->underlinePos + fontPtr->barHeight > fontStructPtr->descent) {
	/*
	 * If this set of cobbled together values would cause the bottom of
	 * the underline bar to stick below the descent of the font, jack
	 * the underline up a bit higher.
	 */

	fontPtr->barHeight = fontStructPtr->descent - fontPtr->underlinePos;
	if (fontPtr->barHeight == 0) {
	    fontPtr->underlinePos--;
	    fontPtr->barHeight = 1;
	}
    }

    return fontPtr;
}

/*
 *---------------------------------------------------------------------------
 *
 * GetControlCharSubst --
 *
 *	When displaying text in a widget, a backslashed escape sequence
 *	is substituted for control characters that occur in the text.
 *	Given a control character, fill in a buffer with the replacement
 *	string that should be displayed.
 *
 * Results:
 *	The return value is the length of the substitute string.  buf is
 *	filled with the substitute string; it is not '\0' terminated.
 *
 * Side effects:
 *	None.
 *
 *---------------------------------------------------------------------------
 */

static int
GetControlCharSubst(c, buf)
    int		c;		/* The control character to be replaced. */
    char	buf[4];		/* Buffer that gets replacement string.  It
				 * only needs to be 4 characters long. */
{
    buf[0] = '\\';
    if ((c < sizeof(mapChars)) && (mapChars[c] != 0)) {
	buf[1] = mapChars[c];
	return 2;
    } else {
	buf[1] = 'x';
	buf[2] = hexChars[(c >> 4) & 0xf];
	buf[3] = hexChars[c & 0xf];
	return 4;
    }
}
